<?php

namespace Drupal\association\Plugin;

use Drupal\association\Entity\AssociationInterface;
use Drupal\association\EntityAdapterManagerInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginBase;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Default base class for association behavior plugins.
 *
 * @see \Drupal\assocation\Plugin\BehaviorBase
 */
abstract class BehaviorBase extends PluginBase implements BehaviorInterface, ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Manager entity type adapters for associating to association entities.
   *
   * @var \Drupal\association\EntityAdapterManagerInterface
   */
  protected $adapterManager;

  /**
   * Create a new instance of the ContentManifestBehavior plugin.
   *
   * @param array $configuration
   *   Plugin configuration options.
   * @param string $plugin_id
   *   The plugin identifier.
   * @param mixed $plugin_definition
   *   The plugin definition (from plugin discovery).
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\association\EntityAdapterManagerInterface $entity_adapter_manager
   *   Manager entity type adapters for associating to association entities.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager, EntityAdapterManagerInterface $entity_adapter_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);

    $this->entityTypeManager = $entity_type_manager;
    $this->adapterManager = $entity_adapter_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('plugin.manager.association.entity_adapter')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getConfiguration() {
    return $this->configuration;
  }

  /**
   * {@inheritdoc}
   */
  public function setConfiguration(array $configuration) {
    $this->configuration = $configuration + $this->defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function getManagerBuilderClass() {
    return $this->pluginDefinition['manager_builder'];
  }

  /**
   * {@inheritdoc}
   */
  public function createAccess(AssociationInterface $association, $tag, $entity_type_id, $bundle, AccountInterface $account): AccessResultInterface {
    $accessResult = $this->isValidEntity($tag, $entity_type_id, $bundle)
      ? AccessResult::allowed()
      : AccessResult::forbidden();

    return $accessResult->addCacheableDependency($association->getType());
  }

  /**
   * {@inheritdoc}
   */
  public function createEntity(AssociationInterface $association, $tag, $entity_type_id, $bundle): ContentEntityInterface {
    if (!$this->isValidEntity($tag, $entity_type_id, $bundle)) {
      $error = sprintf('The bundle %s of entity type %s is not valid for tag %s.', $entity_type_id, $bundle, $tag);
      throw new \InvalidArgumentException($error);
    }

    // Create a new instance of the requested entity type and bundle.
    return $this->adapterManager
      ->getAdapterByEntityType($entity_type_id)
      ->createEntity($bundle);
  }

}
