<?php

namespace Drupal\association\Routing;

use Drupal\association\Controller\AssociationManagementController;
use Drupal\association\EntityAdapterManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Routing\RouteSubscriberBase;
use Drupal\Core\Routing\RoutingEvents;
use Symfony\Component\Routing\Route;
use Symfony\Component\Routing\RouteCollection;

/**
 * Alters Field UI and layout builder routes for association entities.
 */
class AssociationRouteSubscriber extends RouteSubscriberBase {

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity adapter manager.
   *
   * @var \Drupal\association\EntityAdapterManagerInterface
   */
  protected $entityAdapterManager;

  /**
   * Constructs an AssociationRouteSubscriber object.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\association\EntityAdapterManagerInterface $entity_adapter_manager
   *   The entity adapter manager.
   */
  public function __construct(ModuleHandlerInterface $module_handler, EntityTypeManagerInterface $entity_type_manager, EntityAdapterManagerInterface $entity_adapter_manager) {
    $this->moduleHandler = $module_handler;
    $this->entityTypeManager = $entity_type_manager;
    $this->entityAdapterManager = $entity_adapter_manager;
  }

  /**
   * {@inheritdoc}
   */
  protected function alterRoutes(RouteCollection $collection) {
    if ($this->moduleHandler->moduleExists('field_ui')) {
      $this->alterFieldUiRoutes($collection);
    }

    // Add routes for each of the supported entity types to allow linked
    // entities to have a local tasks.
    foreach ($this->entityAdapterManager->getEntityTypes() as $entityTypeId) {
      $entityType = $this->entityTypeManager->getDefinition($entityTypeId);
      $entityRoute = $collection->get("entity.{$entityTypeId}.canonical");
      $linkTemplate = $entityType->getLinkTemplate('canonical');

      // Link doesn't have a canonical link template, or doesn't have a known
      // canonical entity route, we won't have what we need to generate an
      // route to the association management link.
      if (!$linkTemplate && $entityRoute) {
        continue;
      }

      $route = new Route($linkTemplate . '/association');
      $route->addDefaults([
        '_title_callback' => AssociationManagementController::class . '::entityManageTitle',
        '_controller' => AssociationManagementController::class . '::entityManageContent',
        'association_entity_type' => $entityTypeId,
      ]);

      $requirements = [];
      $parameters = $entityRoute->getOption('parameters');

      foreach ($parameters as $name => $paramDef) {
        $requirements[$name] = $entityRoute->getRequirement($name);
      }
      $requirements = array_filter($requirements);
      $requirements['_association_linked_entity_manage_access'] = $entityTypeId;

      // Ensure that user has required permissions to manage the
      // associations's linked content entities.
      $route
        ->setRequirements($requirements)
        ->setOption('parameters', $parameters)
        ->setOption('_admin_route', TRUE);

      $collection->add("association.linked_entity.{$entityTypeId}.manage", $route);
    }
  }

  /**
   * Alter the field_ui and layout_builder routes related to association pages.
   *
   * Association page field_ui and layout_builder routes need additional access
   * checks, and renaming. If the routes are available update them.
   *
   * The access check addition ensures that the field_ui admin is only included
   * for association types that have pages enabled.
   *
   * @param \Symfony\Component\Routing\RouteCollection $collection
   *   The routes collection to alter the routes field_ui from.
   */
  protected function alterFieldUiRoutes(RouteCollection $collection) {
    $entityTypeId = 'association_page';
    $fieldUiRoutes = [
      // Field UI routes.
      "entity.field_config.{$entityTypeId}_field_edit_form",
      "entity.field_config.{$entityTypeId}_storage_edit_form",
      "entity.field_config.{$entityTypeId}_field_delete_form",
      "field_ui.field_storage_config_add_{$entityTypeId}",
      "entity.{$entityTypeId}.field_ui_fields",
      "entity.entity_form_display.{$entityTypeId}.default",
      "entity.entity_form_display.{$entityTypeId}.form_mode",
      "entity.entity_view_display.{$entityTypeId}.default",
      "entity.entity_view_display.{$entityTypeId}.view_mode",
      // Layout builder routes.
      "layout_builder.defaults.{$entityTypeId}.view",
      "layout_builder.defaults.{$entityTypeId}.discard_changes",
      "layout_builder.defaults.{$entityTypeId}.disable",
    ];

    foreach ($fieldUiRoutes as $routeName) {
      if (!($route = $collection->get($routeName))) {
        continue;
      }

      // Routes are only available for pages when the association type has the
      // pages option enabled. Add the access check to the route requirements.
      $route->addRequirements([
        '_association_page_config' => 'association_type',
      ]);

      // Rename these to avoid confusion. Tabs are renamed in the
      // association_local_tasks_alter() hook in association.module file.
      switch ($routeName) {
        case "entity.{$entityTypeId}.field_ui_fields":
          $route->setDefault('_title', 'Page fields');
          break;

        case "entity.entity_form_display.{$entityTypeId}.default":
        case "entity.entity_form_display.{$entityTypeId}.form_mode":
          $route->setDefault('_title', 'Page form display');
          break;

        case "entity.entity_view_display.{$entityTypeId}.default":
        case "entity.entity_view_display.{$entityTypeId}.view_mode":
          $route->setDefault('_title', 'Page display');
          break;
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    // Ensure that this happens alter event triggers after field_ui (-100)
    // and layout_builder (-110) adds their routes.
    $events[RoutingEvents::ALTER][] = ['onAlterRoutes', -200];

    return $events;
  }

}
