<?php

namespace Drupal\association_autogen\Utility;

use Drupal\association\Entity\AssociationInterface;
use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityPublishedInterface;
use Drupal\Core\Entity\EntityStorageException;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\Messenger\MessengerTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Utility\Error;
use Drupal\Core\Utility\Token;
use Drupal\token\TokenEntityMapperInterface;
use Drupal\toolshed\Strategy\Exception\StrategyNotFoundException;

/**
 * The entity generator utility service.
 */
class EntityGenerator implements EntityGeneratorInterface {

  use LoggerChannelTrait;
  use MessengerTrait;

  /**
   * Creates a new instance of the EntityGenerator helper class.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Utility\Token $token
   *   The token replacement service.
   * @param \Drupal\token\TokenEntityMapperInterface $tokenMapper
   *   The entity token mapper service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger channel factory service.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected Token $token,
    protected TokenEntityMapperInterface $tokenMapper,
    LoggerChannelFactoryInterface $loggerFactory,
  ) {
    $this->loggerFactory = $loggerFactory;
  }

  /**
   * {@inheritdoc}
   */
  public function generateMultiple(AssociationInterface $association, array $autogenConfig, bool $saveEntity = TRUE): array {
    $behavior = $association->getBehavior();

    $entities = [];
    foreach ($autogenConfig['generate'] as $autogenId => $genSettings) {
      $tag = $genSettings['tag'];
      [$entityTypeId, $bundleId] = explode(':', $genSettings['entityBundle'] ?? '', 2);
      $entityType = $this->entityTypeManager->getDefinition($entityTypeId, FALSE);

      if ($entityType && $behavior->isValidEntity($tag, $entityTypeId, $bundleId)) {
        $values = [];

        if ($labelKey = $entityType->getKey('label')) {
          $labelPattern = $genSettings['label']['pattern'] ?? '[association:name]: ' . $bundleId;
          $tokenType = $this->tokenMapper->getTokenTypeForEntityType('association', TRUE);

          $values[$labelKey] = $this->token->replace($labelPattern, [$tokenType => $association]);
        }

        try {
          $entity = $behavior->createEntity($association, $tag, $entityTypeId, $bundleId, $values);

          if ($entity instanceof EntityPublishedInterface) {
            $genSettings['active'] ? $entity->setPublished() : $entity->setUnpublished();
          }
          elseif ($statusKey = $entityType->getKey('status')) {
            $entity->set($statusKey, $genSettings['status']);
          }
          if ($saveEntity) {
            $entity->save();
          }

          $link = $association->associateEntity($tag, $entity, FALSE);
          $link->autogen = $autogenId;
          if ($saveEntity) {
            $link->save();
          }

          $entities[] = $entity;
        }
        catch (StrategyNotFoundException | PluginException | \InvalidArgumentException $e) {
          // Either entity adapter, entity type or other field related error
          // occurred. This is likely to mean a module was removed after an
          // entity was configured for auto-generation but the provider has been
          // removed. Log issue, but allow continuing with other auto-saving.
          Error::logException($this->getLogger('association_autogen'), $e);
        }
        catch (EntityStorageException $e) {
          Error::logException($this->getLogger('association_autogen'), $e);

          // Notify user that there was an issue creating this entity.
          $this->messenger()->addError(
            new TranslatableMarkup('Failed to auto-generate entity. Check administrator logs for more details of the error.')
          );
        }
      }
    }

    return $entities;
  }

  /**
   * {@inheritdoc}
   */
  public function generate(AssociationInterface $association, array $autogenerate, bool $saveEntity = TRUE): ?EntityInterface {
    $config = [
      'generate' => [$autogenerate],
    ];

    $entities = $this->generateMultiple($association, $config, $saveEntity);
    return reset($entities);
  }

}
