<?php

namespace Drupal\association_autogen\EventSubscriber;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\association\Entity\AssociationInterface;
use Drupal\association\Entity\AssociationLink;
use Drupal\association\Event\AssociatedEntityFormEvent;
use Drupal\association\Event\AssociationEvents;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Event subscriber to listen for association events.
 *
 * Protected auto-generated entity labels when they are configured to be locked.
 * Preventing the label or title from being changed is only available when an
 * entity for an association tag has a cardinality of one.
 *
 * @see static::onAssociatedFormAlter()
 */
class AssociatedEntitySubscriber implements EventSubscriberInterface {

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      AssociationEvents::INSERT_ASSOCIATED_FORM_ALTER => [
        ['onAssociatedFormAlter', 30],
      ],
      AssociationEvents::UPDATE_ASSOCIATED_FORM_ALTER => [
        ['onAssociatedFormAlter', 30],
      ],
    ];
  }

  /**
   * Alters entity form of content that was auto-generated by associations.
   *
   * Auto-generated content can have a restricted title or other settings that
   * should be adhered to when the auto-generate items are being updated. At the
   * time of this writing, the settings can restrict just the entity labels.
   *
   * @param \Drupal\association\Event\AssociatedEntityFormEvent $event
   *   The form alter event. Provides information about the association and
   *   entity form being altered.
   *
   * @todo Support entity label translations and settings.
   */
  public function onAssociatedFormAlter(AssociatedEntityFormEvent $event): void {
    $assoc = $event->getAssociation();
    $target = $event->getEntity();

    $link = $this->findAssociationLink($assoc, $target);
    $settings = $assoc->getType()->getThirdPartySetting('association_autogen', 'generate', []);
    if ($settings && $link) {
      $autogen = $settings[$link->autogen->value] ?? NULL;

      if ($autogen && empty($autogen['label']['allowEdit'])) {
        if ($labelKey = $target->getEntityType()->getKey('label')) {
          $form = &$event->getForm();

          if (isset($form[$labelKey])) {
            $element = &$form[$labelKey];

            foreach (['widget', 0] as $key) {
              if (isset($element[$key])) {
                $element = &$element[$key];
              }
            }

            foreach ($target->{$labelKey}->first() ?? [] as $property => $value) {
              if (!empty($element[$property]['#type'])) {
                $element[$property]['#value'] = $value->getValue();
                $element[$property]['#disabled'] = TRUE;
              }
            }
          }
        }
      }
    }
  }

  /**
   * Find a matching association link for an entity and association.
   *
   * @param \Drupal\association\Entity\AssociationInterface $association
   *   The association to find a matching link for.
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The content entity to find a link for.
   *
   * @return \Drupal\association\Entity\AssociationLink|null
   *   The association link matching the association and entity. NULL if no
   *   matching association link is available.
   */
  protected function findAssociationLink(AssociationInterface $association, ContentEntityInterface $entity): ?AssociationLink {
    if ($entity->hasField('associations')) {
      /** @var \Drupal\Core\Field\Plugin\Field\FieldType\EntityReferenceItem $item */
      foreach ($entity->get('associations') as $item) {
        /** @var \Drupal\association\Entity\AssociationLink $link */
        $link = $item->entity;

        if ($association->id() == $link->getAssociation()->id()) {
          return $link;
        }
      }
    }

    return NULL;
  }

}
