<?php

namespace Drupal\association_autogen\Form;

use Drupal\Component\Render\FormattableMarkup;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Entity\EntityStorageException;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\Core\Utility\Error;
use Drupal\association\Entity\AssociationTypeInterface;
use Drupal\association\Plugin\BehaviorInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * The settings form for configuring auto-generate entities for associations.
 */
class AssociationAutogenSettingsForm extends FormBase {

  /**
   * Creates a new instance of the AssociationAutogenSettingsForm class.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entityBundleInfo
   *   The entity type bundle info.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EntityTypeBundleInfoInterface $entityBundleInfo,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('entity_type.bundle.info')
    );
  }

  /**
   * The form page title callback.
   *
   * @return \Stringable|string
   *   The form page title to use for this association type.
   */
  public function formTitle(AssociationTypeInterface $association_type): \Stringable|string {
    return $this->t('Edit %association_type_label entity auto-generation settings', [
      '%association_type_label' => $association_type->label(),
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'association_entity_autogen_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?AssociationTypeInterface $association_type = NULL): array {
    if (!$association_type) {
      throw new NotFoundHttpException();
    }

    if ($behavior = $association_type->getBehavior()) {
      $settings = $association_type->getThirdPartySettings('association_autogen');
      if ($form_state->isRebuilding()) {
        $settings = $form_state->getValues() + $settings;
      }

      // Store the association type for use during validation and submission.
      $form_state->set('association_type', $association_type);
      $tags = $behavior->getTags();
      $tagCts = [];
      $tagInfo = [];

      // Get the available tags, and setup a tracker for number of instances.
      foreach ($tags as $tag => $tagLabel) {
        $tagInfo[$tag] = $behavior->getTagInfo($tag);
        $tagCts[$tag] = 0;
      }

      $form['generate'] = [
        '#prefix' => '<div id="entity-generation-wrapper">',
        '#type' => 'table',
        '#empty' => $this->t('empty'),
        '#header' => [
          $this->t('Tag'),
          $this->t('Generate type'),
          $this->t('Published'),
          $this->t('Entity label'),
          $this->t('Operations'),
        ],
      ];

      foreach ($settings['generate'] ?? [] as $id => $autogen) {
        $tag = $autogen['tag'] ?? NULL;

        if (isset($tags[$tag])) {
          $tagCts[$tag]++;

          $form['generate'][$id] = [
            'tag' => [
              '#type' => 'value',
              '#value' => $tag,
              '#plain_text' => $tags[$tag],
            ],
            'entityBundle' => [
              '#type' => 'select',
              '#title' => $this->t('Type of entity to generate'),
              '#title_display' => 'hidden',
              '#options' => $this->getBundleOptions($tagInfo[$tag]['entity_types']),
              '#default_value' => $tags['entityBundle'] ?? NULL,
            ],
            'active' => [
              '#type' => 'checkbox',
              '#title' => $this->t('Publish'),
              '#default_value' => $autogen['active'] ?? FALSE,
            ],
            'label' => [
              'pattern' => [
                '#type' => 'textfield',
                '#title' => $this->t('Auto-generate label pattern'),
                '#title_display' => 'hidden',
                '#default_value' => $autogen['label']['pattern'] ?? $this->t('[association:name]: %tag_label', [
                  '%tag_label' => $tags[$tag],
                ]),
              ],
              'allowEdit' => [
                '#type' => 'checkbox',
                '#title' => $this->t('Allow label edit'),
                '#default_value' => $autogen['label']['allowEdit'] ?? FALSE,
              ],
              'tokens' => [
                '#theme' => 'token_tree_link',
                '#token_types' => ['association', 'node_type'],
              ],
            ],
            'op' => [
              '#type' => 'submit',
              '#value' => $this->t('Remove'),
              '#name' => "remove[$id]",
              '#limit_validation_errors' => [['generate']],
              '#submit' => ['::removeTagSubmit'],
              '#ajax' => [
                'wrapper' => 'entity-generation-wrapper',
                'callback' => '::refreshTagsAjax',
              ],
            ],
          ];
        }
      }

      $available = [];
      foreach ($tags as $tag => $tagLabel) {
        $cardinality = $tagInfo[$tag]['cardinality'];
        if (BehaviorInterface::CARDINALITY_UNLIMITED == $cardinality || $tagCts[$tag] < $cardinality) {
          $available[$tag] = $tagLabel;
        }
      }

      $form['_add_tag'] = [
        '#suffix' => '</div>',
        '#type' => 'fieldset',
        '#title' => $this->t('Add entity'),
        '#tree' => TRUE,
        '#attributes' => [
          'class' => ['container-inline'],
        ],
        'tag' => [
          '#type' => 'select',
          '#title' => $this->t('Add entity tag'),
          '#empty_option' => $this->t('- Select an entity tag -'),
          '#options' => $available,
        ],
        'add' => [
          '#type' => 'submit',
          '#value' => $this->t('Add entity'),
          '#limit_validation_errors' => [
            ['generate'],
            ['_add_tag'],
          ],
          '#validation' => [],
          '#submit' => ['::addTagSubmit'],
          '#ajax' => [
            'wrapper' => 'entity-generation-wrapper',
            'callback' => '::refreshTagsAjax',
          ],
        ],
      ];

      $form['actions']['save'] = [
        '#type' => 'submit',
        '#value' => $this->t('Save'),
        '#attributes' => [
          'class' => ['button', 'button--primary'],
        ],
      ];
    }

    $form['actions']['#type'] = 'actions';
    $form['actions']['cancel'] = [
      '#type' => 'link',
      '#title' => $this->t('Cancel'),
      '#url' => Url::fromRoute('entity.association_type.edit_form', [
        'association_type' => $association_type->id(),
      ]),
      '#attributes' => [
        'class' => ['action-link', 'action-link--danger'],
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    /** @var \Drupal\association\Entity\AssociationTypeInterface $assocType */
    $assocType = $form_state->get('association_type');
    $behavior = $assocType->getBehavior();

    if ($values = &$form_state->getValue('generate')) {
      $counts = [];
      $tagInfo = [];

      foreach ($values as $id => &$autogen) {
        $tag = $autogen['tag'];

        try {
          if (!isset($tagInfo[$tag])) {
            $tagInfo[$tag] = $behavior->getTagInfo($tag);
          }

          $counts[$tag] = ($counts[$tag] ?? 0) + 1;
          $cardinality = $tagInfo[$tag]['cardinality'];
          unset($autogen['op']);

          if (BehaviorInterface::CARDINALITY_UNLIMITED != $cardinality && $counts[$tag] > $cardinality) {
            [$entityTypeId, $bundle] = explode(':', $autogen['entityBundle']);
            $form_state->setError($form['generate'][$id], $this->t('Unable to save auto-generate settings for this %tag_label because it exceeds the limit of %limit.', [
              '%tag_label' => $behavior->getTagLabel($tag, $entityTypeId, $bundle),
              '%limit' => $cardinality,
            ]));
            $form_state->setRebuild(TRUE);
          }
        }
        catch (\InvalidArgumentException $e) {
          // Invalid entity tag, or unsupported.
          unset($values[$id]);
        }
      }
      unset($autogen);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $assocType = $form_state->get('association_type');

    if ($assocType instanceof AssociationTypeInterface) {
      $messenger = $this->messenger();
      /** @var \Drupal\association\Entity\AssociationTypeInterface $assocType */
      $assocType = $this->entityTypeManager
        ->getStorage('association_type')
        ->load($assocType->id());

      try {
        if ($value = $form_state->getValue('generate')) {
          $assocType->setThirdPartySetting('association_autogen', 'generate', $value);
        }
        else {
          $assocType->unsetThirdPartySetting('association_autogen', 'generate');
        }

        $assocType->save();
        $messenger->addStatus($this->t('Saved changes to the %association_type auto-generate entity settings.', [
          '%association_type' => $assocType->label(),
        ]));
      }
      catch (EntityStorageException $e) {
        Error::logException($this->getLogger('association_autogen'), $e);
        $messenger->addError($this->t(
          'An error occurred while trying to save the entity auto-generation settings.'
        ));
      }
    }
  }

  /**
   * Form submit callback to add an entity tag for auto-generation.
   *
   * @param array $form
   *   Reference to the full form structure and elements.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state and values.
   */
  public static function addTagSubmit(array &$form, FormStateInterface $form_state): void {
    if ($addTag = $form_state->getValue(['_add_tag', 'tag'])) {
      $uuid = \Drupal::service('uuid')->generate();

      $generateSettings = $form_state->getValue('generate') ?: [];
      $generateSettings[$uuid] = [
        'tag' => $addTag,
        'label' => [],
      ];
      $form_state->setValue('generate', $generateSettings);
    }

    $form_state->setRebuild();
  }

  /**
   * Form submit callback to remove an entity auto-generate configuration.
   *
   * @param array $form
   *   Reference to the full form structure and elements.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state and structure.
   */
  public static function removeTagSubmit(array &$form, FormStateInterface $form_state): void {
    $trigger = $form_state->getTriggeringElement();

    if (preg_match('#^remove\[([^\]]+)\]$#', $trigger['#name'], $matches)) {
      $values = &$form_state->getValue('generate');
      unset($values[$matches[1]]);
    }

    $form_state->setRebuild();
  }

  /**
   * AJAX callback to refresh the auto-generate table elements.
   *
   * @param array $form
   *   The form structure and elements.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state and values.
   *
   * @return array|\Drupal\Core\Ajax\AjaxResponse
   *   Either renderable array to replace the form elements, or AJAX commands
   *   wrapped in an AjaxResponse.
   */
  public static function refreshTagsAjax(array $form, FormStateInterface $form_state): array|AjaxResponse {
    return [
      $form['generate'],
      $form['_add_tag'],
    ];
  }

  /**
   * Convert the entity type and available bundles into select element options.
   *
   * @param array $entityBundles
   *   A list of entities and bundle combinations to convert into a list of
   *   select options.
   *
   * @return array<string,\Stringable|string>
   *   A list of entity and bundle combinations available for adding that are
   *   derived from the $entityBundles information.
   */
  protected function getBundleOptions(array $entityBundles): array {
    $options = [];
    foreach ($entityBundles as $entityTypeId => $bundles) {
      if ($entityType = $this->entityTypeManager->getDefinition($entityTypeId, FALSE)) {
        if ($entityType->hasKey('bundle')) {
          $bundleInfo = $this->entityBundleInfo->getBundleInfo($entityTypeId);

          foreach ($bundles as $bundle) {
            if (isset($bundleInfo[$bundle])) {
              $options[$entityTypeId . ':' . $bundle] = new FormattableMarkup('@entity_type: %bundle', [
                '@entity_type' => $entityType->getLabel(),
                '%bundle' => $bundleInfo[$bundle]['label'],
              ]);
            }
          }
        }
        else {
          $options[$entityTypeId . ':' . $entityTypeId] = $entityType->getLabel();
        }
      }
    }
    return $options;
  }

}
