<?php

namespace Drupal\at_ls\Controller;

use Drupal\at_ls\Entity\AtlsStringInterface;
use Drupal\at_ls\Service\AtlsApiManagerInterface;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Controller for Asynchronous callback response.
 */
class AtlsAsyncCallbackResponse extends ControllerBase {

  /**
   * The API manager.
   */
  protected AtlsApiManagerInterface $apiManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->apiManager = $container->get('at_ls.api_manager');

    return $instance;
  }

  /**
   * Route response for asynchronous translation callback url.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request object.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
   *   If the token is invalid or not found.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response.
   */
  public function ping(Request $request) {
    // Checks valid token.
    if (!$request->query->has('code')) {
      throw new NotFoundHttpException();
    }
    // Checks the AT-LS token exists.
    $at_ls_token = $request->query->get('code');
    $string_storage = $this->entityTypeManager()->getStorage('at_ls_string');
    $strings = $string_storage->loadByProperties([
      'atls_token' => $at_ls_token,
    ]);
    if (empty($strings)) {
      throw new NotFoundHttpException();
    }

    /** @var \Drupal\at_ls\Entity\AtlsStringInterface $string */
    $string = reset($strings);
    $result = $this->apiManager->getFileByToken($at_ls_token, $string->get('source_language')->value);
    $string->set('status', AtlsStringInterface::ATLS_STATUS_WORKFLOW_STATE[$result['status']]);
    if ($result['base64']) {
      $result = Json::decode(base64_decode($result['base64']));
      if (isset($result['text'])) {
        $string->set('target_text', $result['text']);
      }
    }
    $string->save();
    return new JsonResponse([
      'status' => 'success',
      'message' => 'Ping response from AT-LS.',
      'token' => $at_ls_token,
    ]);
  }

}
