<?php

namespace Drupal\at_ls\Service;

use Drupal\at_ls\Entity\AtlsStringInterface;
use Drupal\at_ls\Entity\AtlsTranslationRequestInterface;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Defines the AT-LS translation request manager service.
 */
class AtlsTranslationRequestManager implements AtlsTranslationRequestManagerInterface {

  use StringTranslationTrait;

  /**
   * AtlsTranslationRequestManager constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\at_ls\Service\AtlsApiManagerInterface $apiManager
   *   The AT-LS API manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The logger channel.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected AtlsApiManagerInterface $apiManager,
    protected ConfigFactoryInterface $configFactory,
    protected LoggerChannelInterface $logger
  ) {}

  /**
   * {@inheritdoc}
   */
  public function buildContent(string $entity_type, string $entity_id, string $source_language): array {
    $content = [];
    $config = $this->configFactory->get('at_ls.mappings');
    /** @var \Drupal\Core\Entity\ContentEntityInterface $entity */
    $entity = $this->entityTypeManager
      ->getStorage($entity_type)
      ->load($entity_id);
    if (!$entity->hasTranslation($source_language)) {
      $this->logger->warning($this->t('The entity @entity_type with ID @entity_id cannot be added to translation request content due it does not have a translation in @source_language.', [
        '@entity_type' => $entity_type,
        '@entity_id' => $entity_id,
        '@source_language' => $source_language,
      ]));
      return $content;
    }
    $entity = $entity->getTranslation($source_language);
    $entity_mappings = $config->get('entity_mappings');
    if (!isset($entity_mappings[$entity_type][$entity->bundle()])) {
      $this->logger->warning($this->t('The @entity_bundle(@entity_type) cannot be added to translation request content due it does not mapped in the mapping configuration.', [
        '@entity_type' => $entity_type,
        '@entity_bundle' => $entity_id,
      ]));
      return $content;
    }
    $fields_to_translate = $entity_mappings[$entity_type][$entity->bundle()];
    $field_definitions = $entity->getFieldDefinitions();

    foreach ($fields_to_translate as $field_name => $properties) {
      if (in_array($field_definitions[$field_name]->getType(), ['entity_reference', 'entity_reference_revisions'])) {
        // Build content to translate from the referenced entities recursively.
        foreach ($entity->get($field_name)->referencedEntities() as $delta => $referenced_entity) {
          $content[$field_name][$delta] = $this->buildContent(
            $referenced_entity->getEntityTypeId(),
            $referenced_entity->id(),
            $source_language
          );
        }
      }
      else {
        foreach ($entity->get($field_name) as $delta => $field) {
          foreach ($properties as $property) {
            if (isset($field->$property) && !empty($field->$property)) {
              $content[$field_name][$delta][$property] = trim($field->$property);
            }
          }
        }
      }
    }

    return $content;
  }

  /**
   * {@inheritdoc}
   */
  public function hasStringsReady(AtlsTranslationRequestInterface $translation_request): bool {
    foreach ($this->getStrings($translation_request) as $string) {
      /** @var \Drupal\at_ls\Entity\AtlsStringInterface $string */
      if (!$string->isReady()) {
        return FALSE;
      }
    }
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function process(AtlsTranslationRequestInterface $translation_request): void {
    if ($translation_request->get('translation_type')->value === AtlsStringInterface::SYNC) {
      $this->processSynchronous($translation_request);
    }
    else {
      $this->processAsynchronous($translation_request);
    }
  }

  /**
   * Encodes source text to Json format.
   *
   * @param string $string
   *   The AT-LS source string.
   *
   * @return string
   *   The json string.
   */
  protected function encodeJsonText(string $string): string {
    return Json::encode(['text' => $string]);
  }

  /**
   * Encodes source text to Json format.
   *
   * @param string $string
   *   The AT-LS source string.
   *
   * @return string
   *   The text.
   */
  protected function decodeJsonText(string $string): string {
    $result = Json::decode($string);
    return $result['text'];
  }

  /**
   * Gets the filename for the string.
   *
   * @param \Drupal\at_ls\Entity\AtlsStringInterface $string
   *   The AT-LS string entity.
   *
   * @return string
   *   The filename.
   */
  protected function getFilename(AtlsStringInterface $string): string {
    // Use the hash as the filename.
    // @todo manage different file extensions.
    return "{$string->get('hash')->value}.json";
  }

  /**
   * Gets AT-LS translation request string.
   *
   * @param \Drupal\at_ls\Entity\AtlsTranslationRequestInterface $translation_request
   *   The AT-LS translation request.
   *
   * @return array
   *   An array of AT-LS strings.
   */
  protected function getStrings(AtlsTranslationRequestInterface $translation_request): array {
    $string_storage = $this->entityTypeManager->getStorage('at_ls_string');
    $strings = $string_storage->loadByProperties([
      'translation_request' => $translation_request->id(),
    ]);

    return $strings;
  }

  /**
   * Process an asynchronous AT-LS translation request.
   *
   * @param \Drupal\at_ls\Entity\AtlsTranslationRequestInterface $translation_request
   *   The AT-LS translation request.
   */
  protected function processAsynchronous(AtlsTranslationRequestInterface $translation_request): void {
    foreach ($this->getStrings($translation_request) as $string) {
      /** @var \Drupal\at_ls\Entity\AtlsStringInterface $string */
      if ($string->get('atls_token')->value) {
        $result = $this->apiManager->getFileByToken(
          $string->get('atls_token')->value,
          $string->get('source_language')->value
        );
        $string->set('status', AtlsStringInterface::ATLS_STATUS_WORKFLOW_STATE[$result['status']]);
        if ($result['base64']) {
          $string->set('target_text', $this->decodeJsonText(base64_decode($result['base64'])));
        }
      }
      else {
        $result = $this->apiManager->translateAsynchronous(
          $string->get('source_language')->value,
          $string->get('target_language')->value,
          $this->getFilename($string),
          $this->encodeJsonText($string->get('source_text')->value),
          $string->getUrlCallback()
        );
        $string->set('atls_token', $result['token']);
      }

      $string->save();
    }
  }

  /**
   * Process a synchronous AT-LS translation request.
   *
   * @param \Drupal\at_ls\Entity\AtlsTranslationRequestInterface $translation_request
   *   The AT-LS translation request.
   */
  protected function processSynchronous(AtlsTranslationRequestInterface $translation_request): void {
    foreach ($this->getStrings($translation_request) as $string) {
      /** @var \Drupal\at_ls\Entity\AtlsStringInterface $string */
      $result = $this->apiManager->translateSynchronous(
        $string->get('source_language')->value,
        $string->get('target_language')->value,
        $this->getFilename($string),
        $this->encodeJsonText($string->get('source_text')->value)
      );

      $string->set('status', AtlsStringInterface::ATLS_STATUS_WORKFLOW_STATE[$result['status']]);
      if ($result['base64']) {
        $string->set('target_text', $this->decodeJsonText(base64_decode($result['base64'])));
      }
      $string->save();
    }
  }

}
