<?php

namespace Drupal\at_ls;

use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Provides common configuration for AT-LS.
 */
trait AtlsTrait {

  use StringTranslationTrait;

  /**
   * Returns the language code.
   *
   * @param string $at_ls_langcode
   *   The AT-LS language code: ISO639-2.
   *
   * @return string
   *   The language code.
   */
  public function getLangCode($at_ls_langcode) {
    return array_search($at_ls_langcode, $this->langCodes());
  }

  /**
   * Returns the AT-LS language code.
   *
   * @param string $lang_code
   *   The language code.
   *
   * @return string
   *   The AT-LS language code: ISO639-2.
   */
  public function getAtlsLangCode($lang_code) {
    return $this->langCodes()[$lang_code];
  }

  /**
   * Returns the language codes mapping.
   *
   * @return array
   *   A mapping array: drupal_langcode => atls_langcode.
   */
  public function langCodes() {
    $langcodes = [];
    $config = $this->configFactory->get('at_ls.languages');
    $language_settings = $config->get('langcodes');

    foreach ($language_settings as $language_setting) {
      $langcodes[$language_setting['drupal_langcode']] = $language_setting['atls_langcode'];
    }

    return $langcodes;
  }

  /**
   * Returns the translation languages.
   *
   * @return array
   *   An array with available translation languages.
   */
  public function getTranslationLanguages() {
    $translation_languages = [];
    $languages = $this->languageManager->getLanguages();

    foreach ($languages as $language) {
      if (isset($this->langCodes()[$language->getId()])) {
        $translation_languages[$language->getId()] = $language->getName();
      }
    }

    return $translation_languages;
  }

}
