<?php

namespace Drupal\at_ls\Controller;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Returns responses for AT-LS module routes.
 */
class AtlsController implements ContainerInjectionInterface {

  use StringTranslationTrait;

  /**
   * Constructs a new AtlsController object.
   *
   * @param \Drupal\Core\Form\FormBuilderInterface $formBuilder
   *   The form builder.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   */
  public function __construct(
    protected FormBuilderInterface $formBuilder,
    protected MessengerInterface $messenger,
    protected Request $request
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('form_builder'),
      $container->get('messenger'),
      $container->get('request_stack')->getCurrentRequest()
    );
  }

  /**
   * Get the form to create AT-LS Translation Request.
   *
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   A RouteMatch object.
   *
   * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
   *   The render form.
   */
  public function getCreateAtlsTranslationRequestForm(RouteMatchInterface $route_match) {
    $entity_type_id = $route_match->getRouteObject()->getOption('_entity_type_id');
    if (empty($entity_type_id)) {
      $this->messenger->addError($this->t('The entity type is not defined in the route object. Please contact the site administrator.'));
      if (!$this->request->query->has('destination')) {
        $url = Url::fromRoute('entity.at_ls_translation_request.collection');
      }
      $url = Url::fromUri($this->request->query->get('destination'));
      return new RedirectResponse($url->toString());
    }

    $entity_id = $route_match->getParameter($entity_type_id);

    return $this->formBuilder->getForm('Drupal\at_ls\Form\CreateAtlsTranslationRequestForm', $entity_type_id, $entity_id);
  }

}
