<?php

namespace Drupal\at_ls\EventSubscriber;

use Drupal\at_ls\Event\AtlsTranslatableEntityEvent;
use Drupal\at_ls\Event\AtlsTranslationRequestEvent;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\TranslatableInterface;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Serializer\Encoder\EncoderInterface;
use Symfony\Component\Serializer\SerializerInterface;

/**
 * Defines AT-LS Translatable Entity event subscriber class.
 */
class AtlsTranslatableEntityEventSubscriber implements EventSubscriberInterface {

  /**
   * Constructs a AtlsTranslatableEntityEventSubscriber object.
   *
   * @param \Symfony\Component\HttpFoundation\RequestStack $request
   *   The request stack.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Symfony\Component\Serializer\SerializerInterface|\Symfony\Component\Serializer\Encoder\EncoderInterface $serializer
   *   The serializer.
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $eventDispatcher
   *   The event dispatcher.
   */
  public function __construct(
    protected RequestStack $request,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected SerializerInterface|EncoderInterface $serializer,
    protected AccountInterface $currentUser,
    protected ConfigFactoryInterface $configFactory,
    protected EventDispatcherInterface $eventDispatcher
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    $events[AtlsTranslatableEntityEvent::INSERT] = ['insert'];
    $events[AtlsTranslatableEntityEvent::UPDATE] = ['update'];
    return $events;
  }

  /**
   * Creates automatically translation request when the entity is inserted.
   *
   * @param \Drupal\at_ls\Event\AtlsTranslatableEntityEvent $event
   *   The translatable entity event.
   */
  public function insert(AtlsTranslatableEntityEvent $event): void {
    $entity = $event->getEntity();
    $config = $this->configFactory->get('at_ls.settings');
    if (!$config->get('automatic_enable')) {
      return;
    }

    $entity_settings = $config->get('automatic_settings');
    // @todo use content_translation.manager service to check entity is translatable.
    if (isset($entity_settings[$entity->getEntityTypeId()][$entity->bundle()])) {
      $content = $this->buildUpdateContent($entity);
      $this->createTranslationRequest($entity, $content);
    }
  }

  /**
   * Creates automatically translation request when the entity is updated.
   *
   * @param \Drupal\at_ls\Event\AtlsTranslatableEntityEvent $event
   *   The translatable entity event.
   */
  public function update(AtlsTranslatableEntityEvent $event): void {
    $entity = $event->getEntity();
    $config = $this->configFactory->get('at_ls.settings');
    if (!$config->get('automatic_enable')) {
      return;
    }

    $entity_settings = $config->get('automatic_settings');
    // @todo use content_translation.manager service to check entity is translatable.
    if ($this->hasChangesOnTranslatableFields($entity) && isset($entity_settings[$entity->getEntityTypeId()][$entity->bundle()])) {
      $content = $this->buildUpdateContent($entity);
      $this->createTranslationRequest($entity, $content);
    }
  }

  /**
   * Builds the content to update of the AT-LS translation request to translate.
   *
   * @param \Drupal\Core\Entity\TranslatableInterface $entity
   *   The entity.
   *
   * @return array
   *   An array with the content to translate.
   */
  protected function buildUpdateContent(TranslatableInterface $entity): array {
    $content = [];
    $config = $this->configFactory->get('at_ls.mappings');
    $entity_type = $entity->getEntityTypeId();
    $entity_mappings = $config->get('entity_mappings');
    $fields_to_translate = $entity_mappings[$entity_type][$entity->bundle()];
    /** @var \Drupal\Core\Entity\ContentEntityInterface $entity */
    $field_definitions = $entity->getFieldDefinitions();

    foreach ($fields_to_translate as $field_name => $properties) {
      if (in_array($field_definitions[$field_name]->getType(), ['entity_reference', 'entity_reference_revisions'])) {
        break;
      }
      foreach ($entity->get($field_name) as $delta => $field) {
        foreach ($properties as $property) {
          $content[$field_name][$delta][$property] = trim($field->$property);
        }
      }
    }

    return $content;
  }

  /**
   * Creates the AT-LS Translation Request.
   *
   * @param \Drupal\Core\Entity\TranslatableInterface $entity
   *   The entity.
   * @param array $content
   *   The the content to translate.
   */
  protected function createTranslationRequest(TranslatableInterface $entity, array $content): void {
    $config = $this->configFactory->get('at_ls.settings');
    $current_request = $this->request->getCurrentRequest();
    $origin_url = $current_request->getSchemeAndHttpHost() . $current_request->getBasePath();
    $entity_type = $entity->getEntityTypeId();
    $entity_id = $entity->id();
    $target_languages = $config->get('automatic_settings.' . $entity_type . '.' . $entity->bundle() . '.target_languages');
    $source_language = $config->get('automatic_settings.' . $entity_type . '.' . $entity->bundle() . '.source_language');
    $translation_type = $config->get('automatic_settings.' . $entity_type . '.' . $entity->bundle() . '.translation_type');

    foreach ($target_languages as $target_language) {
      /** @var \Drupal\at_ls\Entity\AtlsTranslationRequestInterface $translation_request */
      $translation_request = $this->entityTypeManager->getStorage('at_ls_translation_request')->create([
        'uid' => $this->currentUser->id(),
        'origin_url' => $origin_url,
        'entity_type' => $entity_type,
        'entity_id' => $entity_id,
        'source_language' => $source_language,
        'target_language' => $target_language,
        'translation_type' => $translation_type,
        'content' => $this->serializer->encode($content, 'json'),
      ]);
      $translation_request->save();

      $this->eventDispatcher->dispatch(
        new AtlsTranslationRequestEvent($translation_request),
        AtlsTranslationRequestEvent::INSERT
      );
    }
  }

  /**
   * Check changes in the fields with respect to the original entity.
   *
   * @param \Drupal\Core\Entity\TranslatableInterface $entity
   *   The entity.
   *
   * @return bool
   *   TRUE if there are changes in the translatable fields, FALSE otherwise.
   */
  protected function hasChangesOnTranslatableFields(TranslatableInterface $entity): bool {
    $has_changes = FALSE;
    $config = $this->configFactory->get('at_ls.mappings');
    $entity_mappings = $config->get('entity_mappings');
    $entity_type = $entity->getEntityTypeId();
    $entity_bundle = $entity->bundle();

    if (!isset($entity_mappings[$entity_type][$entity_bundle])) {
      return $has_changes;
    }

    $fields_to_translate = $entity_mappings[$entity_type][$entity_bundle];
    /** @var \Drupal\Core\Entity\ContentEntityInterface $entity */
    $field_definitions = $entity->getFieldDefinitions();
    if ($entity->original) {
      $original_entity = $entity->original->toArray();

      foreach ($fields_to_translate as $field_name => $properties) {
        if (in_array($field_definitions[$field_name]->getType(), ['entity_reference', 'entity_reference_revisions'])) {
          break;
        }
        foreach ($entity->get($field_name) as $delta => $field) {
          foreach ($properties as $property) {
            if ($original_entity[$field_name][$delta][$property] !== $field->$property) {
              $has_changes = TRUE;
              break;
            }
          }
        }
      }
    }

    return $has_changes;
  }

}
