<?php

namespace Drupal\at_ls\EventSubscriber;

use Drupal\advancedqueue\Entity\Queue;
use Drupal\advancedqueue\Exception\DuplicateJobException;
use Drupal\advancedqueue\Job;
use Drupal\at_ls\Entity\AtlsString;
use Drupal\at_ls\Entity\AtlsTranslationRequestInterface;
use Drupal\at_ls\Event\AtlsTranslationRequestEvent;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\Serializer\Encoder\DecoderInterface;
use Symfony\Component\Serializer\SerializerInterface;

/**
 * Defines AT-LS Translation Request event subscriber class.
 */
class AtlsTranslationRequestEventSubscriber implements EventSubscriberInterface {

  use StringTranslationTrait;

  /**
   * Constructs a AtlsTranslationRequestEventSubscriber object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The current route match.
   * @param \Symfony\Component\Serializer\SerializerInterface|\Symfony\Component\Serializer\Encoder\DecoderInterface $serializer
   *   The serializer.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The LoggerChannelFactoryInterface object.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected SerializerInterface|DecoderInterface $serializer,
    protected LoggerChannelInterface $logger,
    protected ConfigFactoryInterface $configFactory,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    $events[AtlsTranslationRequestEvent::INSERT] = ['insert'];
    $events[AtlsTranslationRequestEvent::PROCESS] = ['process'];
    return $events;
  }

  /**
   * Inserts the AT-LS Translation Request.
   *
   * @param \Drupal\at_ls\Event\AtlsTranslationRequestEvent $event
   *   The translation request event.
   */
  public function insert(AtlsTranslationRequestEvent $event) {
    $translation_request = $event->getTranslationRequest();

    $this->createJob($translation_request->id());
    $this->createStrings($translation_request);
  }

  /**
   * Process the AT-LS Translation Request.
   *
   * @param \Drupal\at_ls\Event\AtlsTranslationRequestEvent $event
   *   The Translation Request event.
   */
  public function process(AtlsTranslationRequestEvent $event) {
    $translation_request = $event->getTranslationRequest();
    $translation_request->set('attempts', ++$translation_request->get('attempts')->value);
    $translation_request->save();
    /** @var \Drupal\state_machine\Plugin\Field\FieldType\StateItemInterface $state_item */
    $state_item = $translation_request->get('status')->first();

    try {
      $state_item->applyTransitionById('to_processed');
    }
    catch (\Exception $e) {
      throw $e;
    }
    if ($translation_request->get('status')->value === 'processed') {
      try {
        $this->updateEntity($translation_request);
        $translation_request->save();
      }
      catch (\Exception $e) {
        $state_item->applyTransitionById('to_pending');
        throw $e;
      }
    }
  }

  /**
   * Updates the requested entity.
   *
   * @param \Drupal\at_ls\Entity\AtlsTranslationRequestInterface $translation_request
   *   The Translation Request.
   */
  protected function updateEntity(AtlsTranslationRequestInterface $translation_request): void {
    $entity = $this->entityTypeManager
      ->getStorage($translation_request->get('entity_type')->value)
      ->load($translation_request->get('entity_id')->value);
    if ($entity instanceof ContentEntityInterface) {
      $this->updateTranslation(
        $entity,
        $this->serializer->decode($translation_request->get('content')->value, 'json'),
        $translation_request->get('source_language')->value,
        $translation_request->get('target_language')->value
      );
    }
  }

  /**
   * Updates or creates the entity translation.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to save.
   * @param array $content_translation
   *   The content to translate.
   * @param string $source_langcode
   *   The source language code.
   * @param string $target_langcode
   *   The target language code.
   * @param bool $is_referenced_entity
   *   Indicates if the entity is a referenced entity.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  protected function updateTranslation(
    ContentEntityInterface $entity,
    array $content_translation,
    string $source_langcode,
    string $target_langcode,
    bool $is_referenced_entity = FALSE,
  ): bool {
    try {
      static $translated_entities;

      // Do not translate the same entity more than one time.
      if (isset($translated_entities[$entity->getEntityTypeId()][$entity->id()])) {
        return TRUE;
      }
      $translated_entities[$entity->getEntityTypeId()][$entity->id()] = TRUE;

      // Early return if the entity is not translatable.
      if (!$entity->isTranslatable()) {
        return FALSE;
      }

      if ($entity->hasTranslation($target_langcode)) {
        $translation = $entity->getTranslation($target_langcode);
        $original_values = $translation->toArray();
      }
      else {
        $translation = $entity->addTranslation($target_langcode);
        $original_values = $entity->toArray();
        $original_values['content_translation_source'] = $source_langcode;
        $original_values['langcode'] = $target_langcode;
      }

      // Update referenced entities recursively.
      $field_definitions = $translation->getFieldDefinitions();
      $config = $this->configFactory->get('at_ls.mappings');
      $entity_mappings = $config->get('entity_mappings');

      // Get the fields to translate from configuration.
      if (!isset($entity_mappings[$entity->getEntityTypeId()][$entity->bundle()])) {
        return FALSE;
      }
      $fields_to_translate = $entity_mappings[$entity->getEntityTypeId()][$entity->bundle()];

      foreach ($field_definitions as $field_name => $field_definition) {
        if (in_array($field_definition->getType(), ['entity_reference', 'entity_reference_revisions']) && isset($fields_to_translate[$field_name])) {
          $is_referenced_entity = TRUE;
          foreach ($entity->get($field_name)->referencedEntities() as $delta => $referenced_entity) {
            if ($referenced_entity instanceof ContentEntityInterface && isset($content_translation[$field_name][$delta])) {
              if ($field_definition->isTranslatable() && $referenced_entity->language() !== $target_langcode) {
                $referenced_entity = $this->createEntity($referenced_entity, $target_langcode);
                $original_values[$field_name][$delta] = $referenced_entity;
              }
              $this->updateTranslation(
                $referenced_entity,
                $content_translation[$field_name][$delta],
                $source_langcode,
                $target_langcode,
                $is_referenced_entity
              );
            }
          }
        }
      }

      // Set the new values.
      $this->updateValues($translation, $content_translation, $original_values, $source_langcode, $target_langcode, $is_referenced_entity);
    }
    catch (\Exception $exception) {
      $this->logger->error($this->t('The %type with ID %id could not be processed at this moment: %message', [
        '%type' => $entity->getEntityTypeId(),
        '%id' => $entity->id(),
        '%message' => $exception->getMessage(),
      ]));
      return FALSE;
    }

    return TRUE;
  }

  /**
   * Creates a new entity from given entity.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to save.
   * @param string $target_langcode
   *   The target language code.
   *
   * @return \Drupal\Core\Entity\ContentEntityInterface
   *   The new entity.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  protected function createEntity(ContentEntityInterface $entity, string $target_langcode): ContentEntityInterface {
    $new_entity = $entity->createDuplicate();
    $new_entity->set('content_translation_source', LanguageInterface::LANGCODE_NOT_SPECIFIED);
    $new_entity->set('langcode', $target_langcode);
    $new_entity->save();

    return $new_entity;
  }

  /**
   * Creates a job for the translation request.
   *
   * @param int $translation_request_id
   *   The translation request ID.
   *
   * @throws \Drupal\advancedqueue\Exception\DuplicateJobException
   *   In case of duplication, an exception is thrown.
   */
  protected function createJob(int $translation_request_id): void {
    $job = Job::create('translation_request_job', ['id' => $translation_request_id]);
    $queue = Queue::load('translation_requests');

    try {
      $queue->enqueueJob($job);
    }
    catch (DuplicateJobException $e) {
      $this->logger->error($this->t('The translation request with ID %id is already in the queue.', [
        '%id' => $translation_request_id,
      ]));
    }
  }

  /**
   * Creates the strings for the translation request.
   *
   * @param \Drupal\at_ls\Entity\AtlsTranslationRequestInterface $translation_request
   *   The translation request.
   */
  protected function createStrings(AtlsTranslationRequestInterface $translation_request): void {
    $content = $this->serializer->decode($translation_request->get('content')->value, 'json');
    $string_storage = $this->entityTypeManager->getStorage('at_ls_string');

    array_walk_recursive($content, function ($value, $key) use ($string_storage, $translation_request) {
      $source_language = $translation_request->get('source_language')->value;
      $target_language = $translation_request->get('target_language')->value;
      $translation_type = $translation_request->get('translation_type')->value;
      $translation_request_id = $translation_request->id();
      $user = $translation_request->getOwner();
      $hash = AtlsString::getHash($source_language, $target_language, $value);
      $string = $string_storage->loadByProperties([
        'hash' => $hash,
      ]);
      if (count($string) === 0) {
        $string = $string_storage->create([
          'uid' => $user->id(),
          'source_language' => $source_language,
          'target_language' => $target_language,
          'source_text' => $value,
          'translation_request' => $translation_request_id,
          'translation_type' => $translation_type,
        ]);
      }
      else {
        // If the string exists as sync and the translation request is async,
        // we will replace the string with the async translation.
        /** @var \Drupal\at_ls\Entity\AtlsStringInterface $string */
        $string = reset($string);
        if ($string->get('translation_type')->value === AtlsString::SYNC && $translation_request->get('translation_type')->value === AtlsString::ASYNC) {
          $string->set('uid', $user->id());
          $string->set('status', AtlsString::INIT_WORKFLOW_STATE);
          $string->set('target_text', '');
          $string->set('translation_request', $translation_request_id);
          $string->set('translation_type', AtlsString::ASYNC);
        }
      }
      $string->save();
    });
  }

  /**
   * Updates the entity values.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to save.
   * @param array $new_values
   *   The values to translate.
   * @param array $original_values
   *   The original values of the entity.
   * @param string $source_langcode
   *   The source language code.
   * @param string $target_langcode
   *   The target language code.
   * @param bool $is_referenced_entity
   *   Indicates if the entity is a referenced entity.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  protected function updateValues(
    ContentEntityInterface $entity,
    array $new_values,
    array $original_values,
    string $source_langcode,
    string $target_langcode,
    bool $is_referenced_entity
  ): void {
    $string_storage = $this->entityTypeManager->getStorage('at_ls_string');
    // Ensure we don't change the default langcode.
    unset($original_values['default_langcode']);

    // Set only the translatable fields.
    foreach ($original_values as $field_name => $field_values) {
      if ($entity->getFieldDefinition($field_name)->isTranslatable()) {
        $isTranslatableFromAtls = (
          isset($new_values[$field_name]) &&
          !in_array($entity->getFieldDefinition($field_name)->getType(), [
            'entity_reference',
            'entity_reference_revisions',
          ])
        );
        if ($isTranslatableFromAtls) {
          foreach ($new_values[$field_name] as $delta => $new_value) {
            foreach ($new_value as $property => $value) {
              $hash = AtlsString::getHash(
                $source_langcode,
                $target_langcode,
                $value
              );
              $strings = $string_storage->loadByProperties([
                'hash' => $hash,
              ]);
              /** @var \Drupal\at_ls\Entity\AtlsStringInterface $string */
              $string = reset($strings);
              $field_values[$delta][$property] = $string->get('target_text')->value;
            }
          }
        }
      }
      $entity->set($field_name, $field_values);
    }

    $this->entitySave($entity, $is_referenced_entity);
  }

  /**
   * Saves the entity with generate a new revision.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to save.
   * @param bool $is_referenced_entity
   *   Indicates if the entity is a referenced entity.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   *   In case of failures, an exception is thrown.
   */
  protected function entitySave(ContentEntityInterface $entity, bool $is_referenced_entity): void {
    if ($entity->getEntityType()->isRevisionable() && !$is_referenced_entity) {
      $entity->setNewRevision(TRUE);
    }
    $entity->save();
  }

}
