<?php

namespace Drupal\at_ls\Entity;

use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\RevisionableContentEntityBase;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\user\EntityOwnerTrait;

/**
 * Defines the AT-LS translation request entity class.
 *
 * @ContentEntityType(
 *   id = "at_ls_translation_request",
 *   label = @Translation("AT-LS Translation Request"),
 *   label_collection = @Translation("AT-LS Translation Requests"),
 *   label_singular = @Translation("AT-LS translation request"),
 *   label_plural = @Translation("AT-LS translation requests"),
 *   label_count = @PluralTranslation(
 *     singular = "@count AT-LS translation requests",
 *     plural = "@count AT-LS translation requests",
 *   ),
 *   handlers = {
 *     "list_builder" = "Drupal\at_ls\AtlsTranslationRequestListBuilder",
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "views_data" = "Drupal\at_ls\AtlsTranslationRequestViewsData",
 *     "form" = {
 *       "edit" = "Drupal\at_ls\Form\AtlsTranslationRequestForm",
 *       "process" = "Drupal\at_ls\Form\ProcessAtlsTranslationRequestForm",
 *     },
 *     "route_provider" = {
 *       "default" = "Drupal\at_ls\Routing\AtlsTranslationRequestRouteProvider",
 *     },
 *   },
 *   base_table = "at_ls_translation_request",
 *   revision_table = "at_ls_translation_request_revision",
 *   show_revision_ui = TRUE,
 *   admin_permission = "administer at_ls_translation_request",
 *   entity_keys = {
 *     "id" = "id",
 *     "revision" = "revision_id",
 *     "owner" = "uid",
 *   },
 *   revision_metadata_keys = {
 *     "revision_user" = "revision_uid",
 *     "revision_created" = "revision_timestamp",
 *     "revision_log_message" = "revision_log",
 *   },
 *   links = {
 *     "collection" = "/admin/content/translation-request",
 *     "canonical" = "/at-ls-translation-request/{at_ls_translation_request}",
 *     "edit-form" = "/at-ls-translation-request/{at_ls_translation_request}/edit",
 *     "process-form" = "/at-ls-translation-request/{at_ls_translation_request}/process",
 *   },
 * )
 */
final class AtlsTranslationRequest extends RevisionableContentEntityBase implements AtlsTranslationRequestInterface {

  use EntityChangedTrait;
  use EntityOwnerTrait;

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage): void {
    parent::preSave($storage);
    if (!$this->getOwnerId()) {
      // If no owner has been set explicitly, make the anonymous user the owner.
      $this->setOwnerId(0);
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {

    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['origin_url'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Origin url'))
      ->setDescription(t('The origin url.'))
      ->setRequired(TRUE)
      ->setSetting('max_length', 255)
      ->setSetting('text_processing', 0)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE);

    $fields['uid'] = BaseFieldDefinition::create('entity_reference')
      ->setRevisionable(TRUE)
      ->setLabel(t('Author'))
      ->setSetting('target_type', 'user')
      ->setDefaultValueCallback(self::class . '::getDefaultEntityOwner')
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => 60,
          'placeholder' => '',
        ],
        'weight' => 1,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'author',
        'weight' => 1,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['entity_type'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Entity type'))
      ->setDescription(t('The entity type.'))
      ->setRequired(TRUE)
      ->setSetting('max_length', 255)
      ->setSetting('text_processing', 0)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 2,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 2,
      ])
      ->setDisplayConfigurable('form', TRUE);

    $fields['entity_id'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Entity ID'))
      ->setDescription(t('The entity ID.'))
      ->setRequired(TRUE)
      ->setSetting('max_length', 255)
      ->setSetting('text_processing', 0)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 3,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 3,
      ])
      ->setDisplayConfigurable('form', TRUE);

    $fields['source_language'] = BaseFieldDefinition::create('list_string')
      ->setLabel(t('Source Language'))
      ->setDescription(t('The source language.'))
      ->setSetting('allowed_values_function', ['\Drupal\at_ls\Entity\AtlsString', 'getAllowedLanguages'])
      ->setRequired(TRUE)
      ->setSetting('max_length', 255)
      ->setSetting('text_processing', 0)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'options_select',
        'weight' => 4,
      ])
      ->setDisplayConfigurable('form', TRUE);

    $fields['target_language'] = BaseFieldDefinition::create('list_string')
      ->setLabel(t('Target Language'))
      ->setDescription(t('The target language.'))
      ->setSetting('allowed_values_function', ['\Drupal\at_ls\Entity\AtlsString', 'getAllowedLanguages'])
      ->setRequired(TRUE)
      ->setSetting('max_length', 255)
      ->setSetting('text_processing', 0)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 5,
      ])
      ->setDisplayOptions('form', [
        'type' => 'options_select',
        'weight' => 5,
      ])
      ->setDisplayConfigurable('form', TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created Date'))
      ->setDescription(t('The time that the AT-LS translation request was created.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'timestamp',
        'weight' => 6,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Updated Date'))
      ->setDescription(t('The time that the AT-LS translation request was last edited.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'timestamp',
        'weight' => 7,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['attempts'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Attempts'))
      ->setDescription(t('The number of attempts the entity attempted to process.'))
      ->setDefaultValue(0)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'integer',
        'weight' => 8,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['status'] = BaseFieldDefinition::create('state')
      ->setLabel(t('Status'))
      ->setDescription(t('Indicates the status of the translation request.'))
      ->setRevisionable(TRUE)
      ->setRequired(TRUE)
      ->setSetting('max_length', 255)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 9,
      ])
      ->setDisplayOptions('form', [
        'type' => 'options_select',
        'weight' => 9,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setSetting('workflow', static::WORKFLOW_ID);

    $fields['translation_type'] = BaseFieldDefinition::create('list_string')
      ->setLabel(t('Translation Type'))
      ->setDescription(t('The translation type.'))
      ->setSetting('allowed_values_function', ['\Drupal\at_ls\Entity\AtlsString', 'getAllowedTypes'])
      ->setRequired(TRUE)
      ->setSetting('max_length', 255)
      ->setSetting('text_processing', 0)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 10,
      ])
      ->setDisplayOptions('form', [
        'type' => 'options_select',
        'weight' => 10,
      ])
      ->setDisplayConfigurable('form', TRUE);

    $fields['content'] = BaseFieldDefinition::create('json_native')
      ->setLabel(t('Content'))
      ->setDescription(t('The content.'))
      ->setRequired(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'text_default',
        'weight' => 11,
      ])
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'text_textfield',
        'weight' => 11,
      ])
      ->setDisplayConfigurable('form', TRUE);

    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public function isProcessable(): bool {
    return $this->status->value === static::getProcessableState();
  }

  /**
   * {@inheritdoc}
   */
  public static function getProcessableState(): string {
    // @todo Gets this value from the field type.
    return static::INIT_WORKFLOW_STATE;
  }

}
