<?php

namespace Drupal\at_ls\Entity;

use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\RevisionableContentEntityBase;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Url;
use Drupal\user\EntityOwnerTrait;

/**
 * Defines the AT-LS string entity class.
 *
 * @ContentEntityType(
 *   id = "at_ls_string",
 *   label = @Translation("AT-LS String"),
 *   label_collection = @Translation("AT-LS Strings"),
 *   label_singular = @Translation("at-ls string"),
 *   label_plural = @Translation("at-ls strings"),
 *   label_count = @PluralTranslation(
 *     singular = "@count AT-LS strings",
 *     plural = "@count AT-LS strings",
 *   ),
 *   handlers = {
 *     "list_builder" = "Drupal\at_ls\AtlsStringListBuilder",
 *     "views_data" = "Drupal\views\EntityViewsData",
 *     "route_provider" = {
 *       "html" = "Drupal\Core\Entity\Routing\AdminHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "at_ls_string",
 *   revision_table = "at_ls_string_revision",
 *   show_revision_ui = TRUE,
 *   admin_permission = "administer at_ls_string",
 *   entity_keys = {
 *     "id" = "id",
 *     "revision" = "revision_id",
 *     "owner" = "uid",
 *     "hash" = "hash",
 *   },
 *   revision_metadata_keys = {
 *     "revision_user" = "revision_uid",
 *     "revision_created" = "revision_timestamp",
 *     "revision_log_message" = "revision_log",
 *   },
 *   links = {
 *     "collection" = "/admin/content/string",
 *     "canonical" = "/at-ls-string/{at_ls_string}",
 *   },
 * )
 */
final class AtlsString extends RevisionableContentEntityBase implements AtlsStringInterface {

  use EntityChangedTrait;
  use EntityOwnerTrait;

  /**
   * {@inheritdoc}
   */
  public function getUrlCallback(): string {
    $url = Url::fromRoute(
      'at_ls.asynchronous_callback',
      [],
      [
        'query' => [
          'code' => '',
        ],
        'absolute' => TRUE,
      ]
    );
    return $url->toString();
  }

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage): void {
    parent::preSave($storage);
    if (!$this->getOwnerId()) {
      // If no owner has been set explicitly, make the anonymous user the owner.
      $this->setOwnerId(0);
    }
    $hash = static::getHash($this->get('source_language')->value, $this->get('target_language')->value, $this->get('source_text')->value);
    $this->set('hash', $hash);
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {

    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['status'] = BaseFieldDefinition::create('state')
      ->setLabel(t('Status'))
      ->setDescription(t('Indicates the status of the string.'))
      ->setRevisionable(TRUE)
      ->setRequired(TRUE)
      ->setSetting('max_length', 255)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'state_transition_form',
        'weight' => 10,
      ])
      ->setDisplayOptions('form', [
        'type' => 'options_select',
        'weight' => 11,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setSetting('workflow', static::WORKFLOW_ID);

    $fields['uid'] = BaseFieldDefinition::create('entity_reference')
      ->setRevisionable(TRUE)
      ->setLabel(t('Author'))
      ->setSetting('target_type', 'user')
      ->setDefaultValueCallback(self::class . '::getDefaultEntityOwner')
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => 60,
          'placeholder' => '',
        ],
        'weight' => 15,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'author',
        'weight' => 15,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Authored on'))
      ->setDescription(t('The time that the AT-LS string was created.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'datetime_timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the AT-LS string was last edited.'));

    $fields['source_language'] = BaseFieldDefinition::create('list_string')
      ->setLabel(t('Source Language'))
      ->setDescription(t('The source language.'))
      ->setSetting('allowed_values_function', ['\Drupal\at_ls\Entity\AtlsString', 'getAllowedLanguages'])
      ->setRequired(TRUE)
      ->setSetting('max_length', 255)
      ->setSetting('text_processing', 0)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'string',
        'weight' => -5,
      ])
      ->setDisplayOptions('form', [
        'type' => 'options_select',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('form', TRUE);

    $fields['target_language'] = BaseFieldDefinition::create('list_string')
      ->setLabel(t('Target Language'))
      ->setDescription(t('The target language.'))
      ->setSetting('allowed_values_function', ['\Drupal\at_ls\Entity\AtlsString', 'getAllowedLanguages'])
      ->setRequired(TRUE)
      ->setSetting('max_length', 255)
      ->setSetting('text_processing', 0)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'string',
        'weight' => -5,
      ])
      ->setDisplayOptions('form', [
        'type' => 'options_select',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('form', TRUE);

    $fields['source_text'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('Source Text'))
      ->setDescription(t('The source text.'))
      ->setRequired(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'text_default',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'text_textfield',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE);

    $fields['target_text'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('Target Text'))
      ->setDescription(t('The target text.'))
      ->setRevisionable(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'text_default',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'text_textfield',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE);

    $fields['hash'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Hash'))
      ->setDescription(t('The hash.'))
      ->setSetting('max_length', 255)
      ->setSetting('text_processing', 0)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'string',
        'weight' => -5,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('form', TRUE);

    $fields['translation_request'] = BaseFieldDefinition::create('entity_reference')
      ->setRevisionable(TRUE)
      ->setLabel(t('Translation Request'))
      ->setDescription(t('The translation request.'))
      ->setSetting('target_type', 'at_ls_translation_request')
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => 60,
          'placeholder' => '',
        ],
        'weight' => 15,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['atls_token'] = BaseFieldDefinition::create('string')
      ->setLabel(t('AT-LS Token'))
      ->setDescription(t('The AT-LS token.'))
      ->setSetting('max_length', 255)
      ->setSetting('text_processing', 0)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'string',
        'weight' => -5,
      ]);

    $fields['translation_type'] = BaseFieldDefinition::create('list_string')
      ->setLabel(t('Translation Type'))
      ->setDescription(t('The translation type.'))
      ->setRevisionable(TRUE)
      ->setSetting('allowed_values_function', ['\Drupal\at_ls\Entity\AtlsString', 'getAllowedTypes'])
      ->setRequired(TRUE)
      ->setSetting('max_length', 255)
      ->setSetting('text_processing', 0)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 10,
      ])
      ->setDisplayOptions('form', [
        'type' => 'options_select',
        'weight' => 10,
      ])
      ->setDisplayConfigurable('form', TRUE);

    return $fields;
  }

  /**
   * {@inheritdoc}
   *
   * @see \Drupal\at_ls\AtlsTrait
   */
  public static function getAllowedLanguages(): array {
    $translation_languages = [];
    $languages = \Drupal::languageManager()->getLanguages();

    foreach ($languages as $language) {
      $translation_languages[$language->getId()] = $language->getName();
    }

    return $translation_languages;
  }

  /**
   * {@inheritdoc}
   */
  public static function getAllowedTypes(): array {
    return [
      static::SYNC => t('Automatic'),
      static::ASYNC => t('Professional'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function getHash(string $source_language, string $target_language, string $source_text): string {
    return hash('sha256', $source_language . $target_language . $source_text);
  }

  /**
   * {@inheritdoc}
   */
  public function isReady(): bool {
    return $this->status->value === static::READY_WORKFLOW_STATE && !empty($this->target_text->value);
  }

}
