<?php

namespace Drupal\at_ls\Form;

use Drupal\Component\Utility\NestedArray;
use Drupal\content_translation\ContentTranslationManagerInterface;
use Drupal\Core\Entity\ContentEntityTypeInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Dumper;
use Symfony\Component\Yaml\Yaml;

/**
 * Configure atls entity mappings.
 */
class AtlsMappingsForm extends ConfigFormBase {

  /**
   * The content translation manager.
   *
   * @var \Drupal\content_translation\ContentTranslationManagerInterface
   */
  protected ContentTranslationManagerInterface $contentTranslationManager;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected EntityFieldManagerInterface $entityFieldManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The entity type bundle info.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface
   */
  protected EntityTypeBundleInfoInterface $entityTypeBundleInfo;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->contentTranslationManager = $container->get('content_translation.manager');
    $instance->entityFieldManager = $container->get('entity_field.manager');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->entityTypeBundleInfo = $container->get('entity_type.bundle.info');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'at_ls_mappings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['at_ls.mappings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('at_ls.mappings');
    $dumper = new Dumper(2);
    $entity_mappings = $dumper->dump($config->get('entity_mappings'), PHP_INT_MAX, 0, Yaml::DUMP_EXCEPTION_ON_INVALID_TYPE | Yaml::DUMP_MULTI_LINE_LITERAL_BLOCK);

    $form['entity_mappings'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Configuration'),
      '#description' => $this->t('When you make an Entity Reference or Entity Reference Revisions field translatable, two things can happen. First, if the field itself is translatable, a new entity will be created in the target language. Alternatively, if the field is not translatable but the entity is and is configured, a new translation of the referenced entity will be created.'),
      '#default_value' => $entity_mappings,
      '#attributes' => [
        'data-yaml-editor' => 'true',
      ],
    ];
    $form['get_from_entity'] = [
      '#type' => 'details',
      '#title' => $this->t('Get fields from entity'),
      '#attributes' => [
        'id' => ['fields-from-entity-wrapper'],
      ],
    ];
    $form['get_from_entity']['entity'] = [
      '#type' => 'select',
      '#title' => $this->t('Get translatable fields from entity'),
      '#description' => $this->t('Select the entity which you want to get the translatable fields from.'),
      '#options' => $this->getEntityTypes(),
    ];
    $form['get_from_entity']['get'] = [
      '#type' => 'submit',
      '#value' => $this->t('Get fields'),
      '#submit' => [[$this, 'submitGetFields']],
      '#ajax' => [
        'callback' => [$this, 'refreshFields'],
        'wrapper' => 'fields-from-entity-wrapper',
        'effect' => 'fade',
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('at_ls.mappings')
      ->set('entity_mappings', Yaml::parse($form_state->getValue('entity_mappings')))
      ->save(TRUE);

    parent::submitForm($form, $form_state);
  }

  /**
   * Ajax callback function for get the translatable fields from entity.
   *
   * @param array $form
   *   The form where the translatable fields is being included in.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array
   *   The updated element.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function refreshFields(array &$form, FormStateInterface $form_state) {
    $triggering_element = $form_state->getTriggeringElement();

    $element = NestedArray::getValue($form, array_slice($triggering_element['#array_parents'], 0, -1));
    $element['#open'] = TRUE;
    if (($entity = $form_state->getValue('entity')) !== NULL) {
      [$type, $bundle] = explode(':', $entity);
      // Set the indentation to 2 to match Drupal's coding standards.
      $dumper = new Dumper(2);

      $fields[$type][$bundle] = $this->getTranslatableEntityFields($entity);
      $output = "<pre>";
      $output .= $dumper->dump($fields, PHP_INT_MAX, 0, Yaml::DUMP_EXCEPTION_ON_INVALID_TYPE | Yaml::DUMP_MULTI_LINE_LITERAL_BLOCK);
      $output .= "\n</pre>";
      $element['entity']['#suffix'] = $output;
    }

    return $element;
  }

  /**
   * Form submission handler for get the translatable fields from entity.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function submitGetFields(array $form, FormStateInterface &$form_state) {
    // Rebuild the form.
    $form_state->setRebuild();
  }

  /**
   * Prepare #options array for entity types.
   *
   * @return array
   *   The prepared array of entities and bundles.
   */
  protected function getEntityTypes() {
    $types = [];

    foreach ($this->entityTypeManager->getDefinitions() as $entity_id => $entity_type) {
      // Only allow translatable content entities.
      if ($entity_type instanceof ContentEntityTypeInterface) {
        if ($entity_type->getBundleEntityType() !== NULL) {
          foreach ($this->entityTypeBundleInfo->getBundleInfo($entity_id) as $bundle_id => $bundle_type) {
            if ($this->contentTranslationManager->isEnabled($entity_id, $bundle_id)) {
              $types[$entity_type->getLabel()->__toString()][$entity_id . ':' . $bundle_id] = $bundle_type['label'];
            }
          }
        }
        elseif ($this->contentTranslationManager->isEnabled($entity_id)) {
          $types[$entity_type->getLabel()->__toString()][$entity_id . ':' . $entity_id] = $entity_type->getLabel();
        }
      }
    }

    // Sort by entity type id.
    $type_keys = array_keys($types);
    array_multisort($type_keys, SORT_NATURAL, $types);

    return $types;
  }

  /**
   * Returns the translatable fields from the entity.
   *
   * @param string $entity_type_bundle
   *   The entity type with its bundle.
   *
   * @return array
   *   The composed form with the entity type fields.
   */
  protected function getTranslatableEntityFields($entity_type_bundle) {
    $translatable_fields = [];

    [$type, $bundle] = explode(':', $entity_type_bundle);
    /** @var \Drupal\Core\Field\FieldDefinitionInterface[] $fields */
    $fields = $this->entityFieldManager->getFieldDefinitions($type, $bundle);
    foreach ($fields as $field_name => $field) {
      // If the field is not translatable, check if the field is an entity
      // reference field and the target entity is translatable.
      $translatable_er = FALSE;
      if (!$field->isTranslatable() && in_array($field->getType(), ['entity_reference', 'entity_reference_revisions'])) {
        $referenced_entity = $field->getFieldStorageDefinition()->getSetting('target_type');
        if ($this->contentTranslationManager->isEnabled($referenced_entity)) {
          $translatable_er = TRUE;
        }
      }

      // If the field is translatable or the entity reference target entity
      // is translatable, add the field.
      if ($field->isTranslatable() || $translatable_er) {
        $base_field = BaseFieldDefinition::create($field->getType());

        $field_properties = method_exists($field, 'getPropertyDefinitions') ? $field->getPropertyDefinitions() : $base_field->getPropertyDefinitions();
        if (empty($field_properties)) {
          $field_properties = $base_field->getPropertyDefinitions();
        }
        $field_schema = method_exists($field, 'getSchema') ? $field->getSchema() : $base_field->getSchema();

        // Use only properties with schema.
        if (!empty($field_schema['columns'])) {
          $field_properties = array_intersect_key($field_properties, $field_schema['columns']);
        }

        if (!empty($field_properties)) {
          $translatable_fields[$field_name] = [];

          foreach ($field_properties as $property_name => $property) {
            $translatable_fields[$field_name][] = $property_name;
          }
        }
      }
    }

    return $translatable_fields;
  }

}
