<?php

namespace Drupal\at_ls\Form;

use Drupal\at_ls\Event\AtlsTranslationRequestEvent;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Entity\ContentEntityConfirmFormBase;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Form controller for the at-ls translation request entity process form.
 */
class ProcessAtlsTranslationRequestForm extends ContentEntityConfirmFormBase {

  /**
   * The entity AT-LS Translation Request.
   *
   * @var \Drupal\at_ls\Entity\AtlsTranslationRequestInterface
   */
  protected $entity;

  /**
   * The event dispatcher.
   *
   * @var \Symfony\Component\EventDispatcher\EventDispatcherInterface
   */
  protected $eventDispatcher;

  /**
   * Process AT-LS Translation Request confirm form constructor.
   *
   * @param \Drupal\Core\Entity\EntityRepositoryInterface $entity_repository
   *   The entity repository service.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entity_type_bundle_info
   *   The entity type bundle service.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $event_dispatcher
   *   The event dispatcher.
   */
  public function __construct(
    EntityRepositoryInterface $entity_repository,
    EntityTypeBundleInfoInterface $entity_type_bundle_info,
    TimeInterface $time,
    EventDispatcherInterface $event_dispatcher
  ) {
    parent::__construct($entity_repository, $entity_type_bundle_info, $time);
    $this->eventDispatcher = $event_dispatcher;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity.repository'),
      $container->get('entity_type.bundle.info'),
      $container->get('datetime.time'),
      $container->get('event_dispatcher')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'entity_at_ls_translation_request_process_confirm_form';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to force the processed of AT-LS Translation Request %id?', ['%id' => $this->entity->id()]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    /** @var \Drupal\Core\Entity\ContentEntityInterface $entity */
    $entity = $this->getEntity();
    return $entity->toUrl('collection');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return new TranslatableMarkup('Forces the process of the translation request.');
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    /** @var \Drupal\at_ls\Entity\AtlsTranslationRequestInterface $translation_request */
    $translation_request = $this->getEntity();
    try {
      $this->eventDispatcher->dispatch(
        new AtlsTranslationRequestEvent($translation_request),
        AtlsTranslationRequestEvent::PROCESS
      );
    }
    catch (\Exception $e) {
      $this->messenger()->addWarning($this->t('The translation request could not be processed at this moment.'));
    }

    $form_state->setRedirectUrl($translation_request->toUrl('collection'));
  }

}
