<?php

namespace Drupal\at_ls\Plugin\Derivative;

use Drupal\Component\Plugin\Derivative\DeriverBase;
use Drupal\content_translation\ContentTranslationManagerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Plugin\Discovery\ContainerDeriverInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides dynamic local tasks for AT-LS.
 */
class AtlsTranslationLocalTasks extends DeriverBase implements ContainerDeriverInterface {
  use StringTranslationTrait;

  /**
   * The base plugin ID.
   *
   * @var string
   */
  protected $basePluginId;

  /**
   * The content translation manager.
   *
   * @var \Drupal\content_translation\ContentTranslationManagerInterface
   */
  protected $contentTranslationManager;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new AtlsTranslationLocalTasks.
   *
   * @param string $base_plugin_id
   *   The base plugin ID.
   * @param \Drupal\content_translation\ContentTranslationManagerInterface $content_translation_manager
   *   The content translation manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   */
  public function __construct($base_plugin_id, ContentTranslationManagerInterface $content_translation_manager, ConfigFactoryInterface $config_factory) {
    $this->basePluginId = $base_plugin_id;
    $this->contentTranslationManager = $content_translation_manager;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, $base_plugin_id) {
    return new static(
      $base_plugin_id,
      $container->get('content_translation.manager'),
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getDerivativeDefinitions($base_plugin_definition) {
    $mappings = $this->configFactory->get('at_ls.mappings');
    $entity_mappings = $mappings->get('entity_mappings');
    // Create tabs for all possible entity types.
    foreach ($this->contentTranslationManager->getSupportedEntityTypes() as $entity_type_id => $entity_type) {
      // Skip entity types that are not on mapping settings.
      if (!isset($entity_mappings[$entity_type_id])) {
        continue;
      }
      // Find the route name for the translation overview.
      $translation_route_name = "entity.$entity_type_id.at_ls_translate";
      $base_route_name = "entity.$entity_type_id.canonical";
      $this->derivatives[$translation_route_name] = [
        'entity_type' => $entity_type_id,
        'title' => $this->t('AT-LS Translate'),
        'class' => '\Drupal\at_ls\Plugin\Menu\LocalTask\AtlsTranslationLocalTask',
        'route_name' => $translation_route_name,
        'base_route' => $base_route_name,
      ] + $base_plugin_definition;
    }
    return parent::getDerivativeDefinitions($base_plugin_definition);
  }

}
