<?php

namespace Drupal\at_ls\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure atls module settings.
 */
class AtlsLanguagesForm extends ConfigFormBase {

  /**
   * Config settings.
   *
   * @var string
   */
  const SETTINGS = 'at_ls.languages';

  /**
   * The language manager service.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->languageManager = $container->get('language_manager');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'atls_languages';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      static::SETTINGS,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config(static::SETTINGS);

    $form['langcodes'] = [
      '#type' => 'element_multiple',
      '#title' => $this->t('Language codes'),
      '#header' => TRUE,
      '#sorting' => FALSE,
      '#add_more_input' => FALSE,
      '#empty_items' => 0,
      '#element' => [
        'drupal_langcode' => [
          '#type' => 'select',
          '#title' => $this->t('Drupal langcode'),
          '#options' => $this->getDrupalLangcodes(),
          '#sort_options' => TRUE,
          '#empty_option' => $this->t('- Select a Drupal langcode -'),
        ],
        'atls_langcode' => [
          '#type' => 'select',
          '#title' => $this->t('AT-LS langcode'),
          '#options' => $this->getAtlsLangcodes(),
          '#sort_options' => TRUE,
          '#empty_option' => $this->t('- Select a AT-LS langcode -'),
        ],
      ],
      '#default_value' => $config->get('langcodes'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $language_settings = $form_state->getValue('langcodes');
    $copy_language_settings = $language_settings;

    // Checks if any combination is duplicated.
    foreach ($language_settings as $key => $language_setting) {
      unset($copy_language_settings[$key]);
      foreach ($copy_language_settings as $copy_language_setting) {
        if ($language_setting['drupal_langcode'] === $copy_language_setting['drupal_langcode'] && $language_setting['atls_langcode'] === $copy_language_setting['atls_langcode']) {
          $form_state->setErrorByName('langcodes', $this->t('There is a language codes combination is duplicated.'));
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config(static::SETTINGS)
      ->set('langcodes', $form_state->getValue('langcodes'))
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Gets AT-LS langcodes. Basically ISO639-2 codes.
   *
   * @return array
   *   An array with AT-LS langcodes.
   */
  protected function getAtlsLangcodes() {
    // Iso codes from https://www.loc.gov/standards/iso639-2/php/code_list.php.
    $iso_codes = [
      'aar',
      'abk',
      'afr',
      'aka',
      'amh',
      'ara',
      'arg',
      'arm',
      'asm',
      'ava',
      'ave',
      'aym',
      'aze',
      'bak',
      'bam',
      'baq',
      'bel',
      'ben',
      'bih',
      'bis',
      'bos',
      'bre',
      'bul',
      'bur',
      'cat',
      'cha',
      'che',
      'chi',
      'chu',
      'chv',
      'cor',
      'cos',
      'cre',
      'cze',
      'dan',
      'deu',
      'div',
      'dut',
      'dzo',
      'ell',
      'eng',
      'epo',
      'est',
      'ewe',
      'fao',
      'fij',
      'fin',
      'fra',
      'fre',
      'fry',
      'ful',
      'geo',
      'ger',
      'gla',
      'gle',
      'glg',
      'glv',
      'gre',
      'grn',
      'guj',
      'hat',
      'hau',
      'heb',
      'her',
      'hin',
      'hmo',
      'hrv',
      'hun',
      'ibo',
      'ice',
      'ido',
      'iii',
      'iku',
      'ile',
      'ina',
      'ind',
      'ipk',
      'ita',
      'jav',
      'jpn',
      'kal',
      'kan',
      'kas',
      'kau',
      'kaz',
      'khm',
      'kik',
      'kin',
      'kir',
      'kom',
      'kon',
      'kor',
      'kua',
      'kur',
      'lao',
      'lat',
      'lav',
      'lim',
      'lin',
      'lit',
      'ltz',
      'lub',
      'lug',
      'mac',
      'mah',
      'mal',
      'mao',
      'mar',
      'may',
      'mlg',
      'mlt',
      'mon',
      'nau',
      'nav',
      'nbl',
      'nde',
      'ndo',
      'nep',
      'nld',
      'nno',
      'nob',
      'nor',
      'nya',
      'oci',
      'oji',
      'ori',
      'orm',
      'oss',
      'pan',
      'per',
      'pli',
      'pol',
      'por',
      'pus',
      'que',
      'roh',
      'ron',
      'rum',
      'run',
      'rus',
      'sag',
      'san',
      'sin',
      'slo',
      'slv',
      'sme',
      'smo',
      'sna',
      'snd',
      'som',
      'sot',
      'spa',
      'srd',
      'srp',
      'ssw',
      'sun',
      'swa',
      'swe',
      'tah',
      'tam',
      'tat',
      'tel',
      'tgk',
      'tgl',
      'tha',
      'tib',
      'tir',
      'ton',
      'tsn',
      'tso',
      'tuk',
      'tur',
      'twi',
      'uig',
      'ukr',
      'urd',
      'uzb',
      'ven',
      'vie',
      'vol',
      'wel',
      'wln',
      'wol',
      'xho',
      'yid',
      'yor',
      'zha',
      'zho',
      'zul',
    ];
    // Extra iso codes added by AT-LS.
    $extra_codes = [
      'ame',
      'brz',
      'enc',
      'flm',
      'his',
      'mex',
      'val',
      'zht',
    ];
    $atls_langcodes = array_merge($iso_codes, $extra_codes);

    // @todo Improves this making it more human readable using labels.
    return array_combine($atls_langcodes, $atls_langcodes);
  }

  /**
   * Gets installed language codes.
   *
   * @return array
   *   An array with installed langcodes.
   */
  protected function getDrupalLangcodes() {
    $drupal_langcodes = [];

    foreach ($this->languageManager->getLanguages() as $language) {
      $drupal_langcodes[] = $language->getId();
    }

    return array_combine($drupal_langcodes, $drupal_langcodes);
  }

}
