<?php

namespace Drupal\at_ls\Form;

use Drupal\at_ls\AtlsTrait;
use Drupal\at_ls\Entity\AtlsString;
use Drupal\content_translation\ContentTranslationManagerInterface;
use Drupal\Core\Entity\ContentEntityTypeInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Render\Markup;
use Drupal\Core\Url;
use Drupal\key\KeyRepositoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure atls module settings.
 */
class AtlsSettingsForm extends ConfigFormBase {

  use AtlsTrait;

  /**
   * Config settings.
   *
   * @var string
   */
  const SETTINGS = 'at_ls.settings';

  /**
   * The key repository.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  protected KeyRepositoryInterface $keyRepository;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The content translation manager.
   *
   * @var \Drupal\content_translation\ContentTranslationManagerInterface
   */
  protected ContentTranslationManagerInterface $contentTranslationManager;

  /**
   * The entity type bundle info.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface
   */
  protected EntityTypeBundleInfoInterface $entityTypeBundleInfo;

  /**
   * The language manager service.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->keyRepository = $container->get('key.repository');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->contentTranslationManager = $container->get('content_translation.manager');
    $instance->entityTypeBundleInfo = $container->get('entity_type.bundle.info');
    $instance->languageManager = $container->get('language_manager');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'atls_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      static::SETTINGS,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config(static::SETTINGS);
    $info = $this->t(
      'In order to configure this settings beforehand, it is necessary to previously configure the <a href="@languages-page">language mapping</a> and also create the <a href="@key-page">AT-LS API keys</a>.',
      [
        '@languages-page' => Url::fromRoute('at_ls.languages_form')->toString(),
        '@key-page' => Url::fromRoute('entity.key.collection')->toString(),
      ]
    );
    $html = '<p style="color:red;">' . $info . '</p>';

    $form['info'] = [
      '#markup' => Markup::create($html),
    ];
    $form['api_keys'] = [
      '#type' => 'element_multiple',
      '#title' => $this->t('API Keys'),
      '#header' => TRUE,
      '#sorting' => FALSE,
      '#add_more_input' => FALSE,
      '#empty_items' => 0,
      '#required' => TRUE,
      '#element' => [
        'source_language' => [
          '#type' => 'select',
          '#title' => $this->t('Source language'),
          '#options' => $this->getTranslationLanguages(),
          '#sort_options' => TRUE,
          '#empty_option' => $this->t('- Select a source language -'),
          '#required' => TRUE,
        ],
        'api_key' => [
          '#type' => 'key_select',
          '#title' => $this->t('API key'),
          '#key_type' => 'at_ls',
        ],
      ],
      '#default_value' => $config->get('api_keys'),
    ];
    $form['notification_email'] = [
      '#type' => 'email',
      '#title' => $this->t('Notification email'),
      '#description' => $this->t('The email account to receive notifications from AT-LS service when the asynchronous callback fails.'),
      '#default_value' => $config->get('notification_email') ?? $this->config('system.site')->get('mail'),
    ];
    $form['language_settings'] = [
      '#type' => 'element_multiple',
      '#title' => $this->t('Language settings'),
      '#header' => TRUE,
      '#sorting' => FALSE,
      '#add_more_input' => FALSE,
      '#empty_items' => 0,
      '#required' => TRUE,
      '#element' => [
        'source_language' => [
          '#type' => 'select',
          '#title' => $this->t('Source language'),
          '#options' => $this->getTranslationLanguages(),
          '#sort_options' => TRUE,
          '#empty_option' => $this->t('- Select a source language -'),
          '#required' => TRUE,
        ],
        'target_language' => [
          '#type' => 'select',
          '#title' => $this->t('Target language'),
          '#options' => $this->getTranslationLanguages(),
          '#sort_options' => TRUE,
          '#empty_option' => $this->t('- Select a target language -'),
          '#required' => TRUE,
        ],
      ],
      '#default_value' => $config->get('language_settings'),
    ];
    $form['automatic_translation'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Automatic Translation'),
    ];
    $form['automatic_translation']['automatic_enable'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Automatic Translation'),
      '#description' => $this->t('Enable the automatic translation to create the translation request when products and menu items are inserted.'),
      '#default_value' => $config->get('automatic_enable'),
    ];

    $languages = $this->getTranslationLanguages();

    foreach ($this->getConfigurableEntities() as $entity_type_id => $entity_data) {
      $form['automatic_translation'][$entity_type_id] = [
        '#type' => 'details',
        '#title' => $entity_data['label'],
        '#description' => $this->t('Select the entity which you want to set automatic translation.'),
        '#open' => $config->get('automatic_enable'),
        '#states' => [
          'visible' => [
            ':input[name="automatic_enable"]' => ['checked' => TRUE],
          ],
        ],
      ];
      foreach ($entity_data['bundles'] as $bundle_id => $bundle_label) {
        $element_id = $this->getElementFormId($entity_type_id, $bundle_id);
        $form['automatic_translation'][$entity_type_id][$element_id] = [
          '#type' => 'checkbox',
          '#title' => $bundle_label,
          '#default_value' => !empty($config->get('automatic_settings.' . $entity_type_id . '.' . $bundle_id)) ? TRUE : FALSE,
        ];
        $form['automatic_translation'][$entity_type_id]['fieldset:' . $element_id] = [
          '#type' => 'fieldset',
          '#states' => [
            'visible' => [
              ':input[name="' . $element_id . '"]' => ['checked' => TRUE],
            ],
          ],
        ];
        $form['automatic_translation'][$entity_type_id]['fieldset:' . $element_id]['operations:' . $element_id] = [
          '#type' => 'checkboxes',
          '#title' => $this->t('Operation'),
          '#options' => [
            'insert' => $this->t('Insert'),
            'update' => $this->t('Update'),
          ],
          '#default_value' => $config->get('automatic_settings.' . $entity_type_id . '.' . $bundle_id . '.operations') ?: [],
          '#states' => [
            'required' => [
              ':input[name="' . $element_id . '"]' => ['checked' => TRUE],
            ],
          ],
        ];
        $form['automatic_translation'][$entity_type_id]['fieldset:' . $element_id]['source_language:' . $element_id] = [
          '#type' => 'select',
          '#title' => $this->t('Source language'),
          '#options' => $languages,
          '#default_value' => $config->get('automatic_settings.' . $entity_type_id . '.' . $bundle_id . '.source_language') ?: '',
          '#states' => [
            'required' => [
              ':input[name="' . $element_id . '"]' => ['checked' => TRUE],
            ],
          ],
        ];
        $form['automatic_translation'][$entity_type_id]['fieldset:' . $element_id]['target_languages:' . $element_id] = [
          '#type' => 'checkboxes',
          '#title' => $this->t('Target languages'),
          '#multiple' => TRUE,
          '#options' => $languages,
          '#default_value' => $config->get('automatic_settings.' . $entity_type_id . '.' . $bundle_id . '.target_languages') ?: [],
          '#states' => [
            'required' => [
              ':input[name="' . $element_id . '"]' => ['checked' => TRUE],
            ],
          ],
        ];
        $form['automatic_translation'][$entity_type_id]['fieldset:' . $element_id]['translation_type:' . $element_id] = [
          '#type' => 'select',
          '#title' => $this->t('Type'),
          '#options' => AtlsString::getAllowedTypes(),
          '#default_value' => $config->get('automatic_settings.' . $entity_type_id . '.' . $bundle_id . '.translation_type'),
          '#states' => [
            'required' => [
              ':input[name="' . $element_id . '"]' => ['checked' => TRUE],
            ],
          ],
        ];
      }
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    // Validate language settings.
    $api_keys = $form_state->getValue('api_keys');
    $language_settings = $form_state->getValue('language_settings');
    foreach ($language_settings as $key => $language_setting) {
      if ($language_setting['source_language'] === $language_setting['target_language']) {
        $form_state->setErrorByName('language_settings', $this->t('The source language cannot be selected as a target language.'));
      }

      if (!empty($api_keys)) {
        $found = FALSE;
        foreach ($api_keys as $api_key) {
          if ($language_setting['source_language'] === $api_key['source_language']) {
            $found = TRUE;
            break;
          }
        }
        if (!$found) {
          $form_state->setErrorByName('language_settings', $this->t('The source language is not available in the API keys.'));
        }
      }
    }

    $copy_language_settings = $language_settings;
    // Checks if a combination is duplicated.
    foreach ($language_settings as $key => $language_setting) {
      unset($copy_language_settings[$key]);
      foreach ($copy_language_settings as $copy_language_setting) {
        if ($language_setting['source_language'] === $copy_language_setting['source_language'] && $language_setting['target_language'] === $copy_language_setting['target_language']) {
          $form_state->setErrorByName('language_settings', $this->t('There is a language combination duplicated.'));
        }
      }
    }

    // Validate automatic translation settings.
    foreach ($this->getConfigurableEntities() as $entity_type_id => $entity_data) {
      foreach ($entity_data['bundles'] as $bundle_id => $bundle_label) {
        $element_id = $this->getElementFormId($entity_type_id, $bundle_id);
        $source_language = $form_state->getValue('source_language:' . $element_id);
        $target_languages = array_filter($form_state->getValue('target_languages:' . $element_id));
        if (array_key_exists($source_language, $target_languages)) {
          $form_state->setErrorByName('target_languages:' . $element_id, $this->t('The source language cannot be selected as a target language.'));
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config(static::SETTINGS)
      ->set('api_keys', $form_state->getValue('api_keys'))
      ->set('notification_email', $form_state->getValue('notification_email'))
      ->set('language_settings', $form_state->getValue('language_settings'))
      ->set('automatic_enable', $form_state->getValue('automatic_enable'))
      ->set('automatic_settings', $this->getAutomaticTranslationSettings($form_state))
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Get automatic translation settings for commerce_product.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   An associative array of automatic translation settings.
   */
  protected function getAutomaticTranslationSettings(FormStateInterface $form_state) {
    $automatic_translation = [];

    foreach ($this->getConfigurableEntities() as $entity_id => $entity_data) {
      foreach ($entity_data['bundles'] as $bundle_id => $bundle_label) {
        $element_id = $this->getElementFormId($entity_id, $bundle_id);
        if ($form_state->getValue($element_id)) {
          $automatic_translation[$entity_id][$bundle_id] = [
            'operations' => array_filter($form_state->getValue('operations:' . $element_id)),
            'source_language' => $form_state->getValue('source_language:' . $element_id),
            'target_languages' => array_filter($form_state->getValue('target_languages:' . $element_id)),
            'translation_type' => $form_state->getValue('translation_type:' . $element_id),
          ];
        }
      }
    }

    return $automatic_translation;
  }

  /**
   * Get configurable entities.
   *
   * @return array
   *   An associative array with data of configurable entities.
   */
  protected function getConfigurableEntities() {
    $configurable_entities = [];
    foreach ($this->entityTypeManager->getDefinitions() as $entity_id => $entity_type) {
      // Only allow translatable content entities.
      if ($entity_type instanceof ContentEntityTypeInterface) {
        if ($entity_type->getBundleEntityType() !== NULL) {
          $bundles = [];
          foreach ($this->entityTypeBundleInfo->getBundleInfo($entity_id) as $bundle_id => $bundle_type) {
            if ($this->contentTranslationManager->isEnabled($entity_id, $bundle_id)) {
              $bundles[$bundle_id] = $bundle_type['label'];
            }
          }
          if (!empty($bundles)) {
            $configurable_entities[$entity_id] = [
              'label' => $entity_type->getLabel()->__toString(),
              'bundles' => $bundles,
            ];
          }
        }
      }
    }

    // Sort by entity type id.
    $configurable_entities_keys = array_keys($configurable_entities);
    array_multisort($configurable_entities_keys, SORT_NATURAL, $configurable_entities);

    return $configurable_entities;
  }

  /**
   * Get element form ID.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   *
   * @return string
   *   The element form ID.
   */
  protected function getElementFormId(string $entity_type_id, string $bundle_id) {
    return $entity_type_id . ':' . $bundle_id;
  }

}
