<?php

namespace Drupal\at_ls\Form;

use Drupal\at_ls\AtlsTrait;
use Drupal\at_ls\Entity\AtlsString;
use Drupal\at_ls\Event\AtlsTranslationRequestEvent;
use Drupal\at_ls\Service\AtlsTranslationRequestManagerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Serializer\Encoder\EncoderInterface;
use Symfony\Component\Serializer\SerializerInterface;

/**
 * Create AT-LS translation request confirm form.
 */
class CreateAtlsTranslationRequestForm extends ConfirmFormBase {

  use AtlsTrait;

  /**
   * The current request.
   *
   * @var \Symfony\Component\HttpFoundation\Request
   */
  protected $request;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The serializer.
   *
   * @var \Symfony\Component\Serializer\SerializerInterface|\Symfony\Component\Serializer\Encoder\EncoderInterface
   */
  protected $serializer;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The event dispatcher.
   *
   * @var \Symfony\Component\EventDispatcher\EventDispatcherInterface
   */
  protected $eventDispatcher;

  /**
   * The at_ls logging channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * The language manager service.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The AT-LS translation request manager.
   *
   * @var \Drupal\at_ls\Service\AtlsTranslationRequestManagerInterface
   */
  protected $translationRequestManager;

  /**
   * CreateAtlsTranslationRequestForm constructor.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Symfony\Component\Serializer\SerializerInterface|\Symfony\Component\Serializer\Encoder\EncoderInterface $serializer
   *   The serializer service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $event_dispatcher
   *   The event dispatcher.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The at_ls logging channel.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\at_ls\Service\AtlsTranslationRequestManagerInterface $translation_request_manager
   *   The AT-LS translation request manager.
   */
  public function __construct(
    Request $request,
    MessengerInterface $messenger,
    SerializerInterface|EncoderInterface $serializer,
    EntityTypeManagerInterface $entity_type_manager,
    EventDispatcherInterface $event_dispatcher,
    LoggerChannelInterface $logger,
    LanguageManagerInterface $language_manager,
    ConfigFactoryInterface $config_factory,
    AtlsTranslationRequestManagerInterface $translation_request_manager
  ) {
    $this->request = $request;
    $this->messenger = $messenger;
    $this->serializer = $serializer;
    $this->entityTypeManager = $entity_type_manager;
    $this->eventDispatcher = $event_dispatcher;
    $this->logger = $logger;
    $this->languageManager = $language_manager;
    $this->configFactory = $config_factory;
    $this->translationRequestManager = $translation_request_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('request_stack')->getCurrentRequest(),
      $container->get('messenger'),
      $container->get('serializer'),
      $container->get('entity_type.manager'),
      $container->get('event_dispatcher'),
      $container->get('logger.factory')->get('at_ls'),
      $container->get('language_manager'),
      $container->get('config.factory'),
      $container->get('at_ls.translation_request.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'create_at_ls_translation_request_confirm_form';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to send this to AT-LS get a translation?');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    if (!$this->request->query->has('destination')) {
      return Url::fromRoute('entity.at_ls_translation_request.collection');
    }
    return Url::fromUri($this->request->query->get('destination'));
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return $this->t('Sends a translation request to AT-LS for the selected entity.');
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $entity_type = NULL, $entity_id = NULL) {
    $form = parent::buildForm($form, $form_state);
    $form['entity_type'] = [
      '#type' => 'hidden',
      '#value' => $entity_type,
    ];
    $form['entity_id'] = [
      '#type' => 'hidden',
      '#value' => $entity_id,
    ];
    $source_language = $this->getSourceLanguage($entity_type, $entity_id);
    $form['source_language'] = [
      '#type' => 'hidden',
      '#value' => $source_language,
    ];

    // Get the target languages available for translation.
    $target_languages = $this->getTranslationLanguages();
    $config = $this->configFactory->get('at_ls.settings');
    $language_settings = $config->get('language_settings');

    foreach ($target_languages as $key => $target_language) {
      $found = FALSE;
      foreach ($language_settings as $language_setting) {
        if ($key === $language_setting['target_language'] && $source_language === $language_setting['source_language']) {
          $found = TRUE;
        }
      }
      if (!$found) {
        unset($target_languages[$key]);
      }
    }
    $form['target_language'] = [
      '#type' => 'checkboxes',
      '#required' => TRUE,
      '#title' => $this->t('Select target language'),
      '#options' => $target_languages,
      '#multiple' => TRUE,
    ];
    $form['translation_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Select translation type'),
      '#options' => AtlsString::getAllowedTypes(),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $config = $this->configFactory->get('at_ls.settings');
    $language_settings = $config->get('language_settings');
    $source_language = $form_state->getValue('source_language');
    $target_languages = array_filter($form_state->getValue('target_language'));

    foreach ($target_languages as $target_language) {
      $found = FALSE;
      foreach ($language_settings as $language_setting) {
        if ($source_language === $language_setting['source_language'] && $target_language === $language_setting['target_language']) {
          $found = TRUE;
          break;
        }
      }
      if (!$found) {
        $form_state->setErrorByName('target_language', $this->t('The selected target language is not available for translation.'));
        return;
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->createTranslationRequests($form_state);

    if (!$this->request->query->has('destination')) {
      $entity = $this->entityTypeManager
        ->getStorage($form_state->getValue('entity_type'))
        ->load($form_state->getValue('entity_id'));
      $url = $entity->toUrl();
    }
    else {
      $url = Url::fromUserInput($this->request->query->get('destination'));
    }
    $this->messenger()->addStatus($this->t('The translation request has been sent.'));
    $form_state->setRedirectUrl($url);
  }

  /**
   * Creates the AT-LS Translation Request entities.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  protected function createTranslationRequests(FormStateInterface $form_state): void {
    $entity_type = $form_state->getValue('entity_type');
    $entity_id = $form_state->getValue('entity_id');
    $source_language = $form_state->getValue('source_language');
    $content = $this->translationRequestManager->buildContent($entity_type, $entity_id, $source_language);

    // Create a translation request for each target language.
    foreach ($form_state->getValue('target_language') as $target_language) {
      if ($target_language) {
        /** @var \Drupal\at_ls\Entity\AtlsTranslationRequestInterface $translation_request */
        $translation_request = $this->entityTypeManager->getStorage('at_ls_translation_request')->create([
          'uid' => $this->currentUser()->id(),
          'origin_url' => $this->request->getSchemeAndHttpHost() . $this->request->getBasePath(),
          'entity_type' => $entity_type,
          'entity_id' => $entity_id,
          'source_language' => $source_language,
          'target_language' => $target_language,
          'translation_type' => $form_state->getValue('translation_type'),
          'content' => $this->serializer->encode($content, 'json'),
        ]);
        $translation_request->save();

        $this->eventDispatcher->dispatch(
          new AtlsTranslationRequestEvent($translation_request),
          AtlsTranslationRequestEvent::INSERT
        );
      }
    }
  }

  /**
   * Gets the source language.
   *
   * @param string $entity_type
   *   The entity type.
   * @param int $entity_id
   *   The entity ID.
   *
   * @return string
   *   The source language.
   */
  protected function getSourceLanguage(string $entity_type, int $entity_id): string {
    $entity = $this->entityTypeManager->getStorage($entity_type)->load($entity_id);
    return $entity->language()->getId();
  }

}
