<?php

namespace Drupal\at_ls\Plugin\AdvancedQueue\JobType;

use Drupal\advancedqueue\Job;
use Drupal\advancedqueue\JobResult;
use Drupal\advancedqueue\Plugin\AdvancedQueue\JobType\JobTypeBase;
use Drupal\at_ls\Entity\AtlsStringInterface;
use Drupal\at_ls\Event\AtlsTranslationRequestEvent;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * AT-LS Translation Request job type.
 *
 * @AdvancedQueueJobType(
 *   id = "translation_request_job",
 *   label = @Translation("Translation Request Job"),
 *   allow_duplicates = false,
 * )
 */
class AtlsTranslationRequestJob extends JobTypeBase implements ContainerFactoryPluginInterface {

  /**
   * Entity manager service.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The event dispatcher.
   */
  protected EventDispatcherInterface $eventDispatcher;

  /**
   * Logger factory.
   */
  protected LoggerChannelInterface $logger;

  /**
   * {@inheritdoc}
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    EntityTypeManagerInterface $entity_type_manager,
    EventDispatcherInterface $event_dispatcher,
    LoggerChannelInterface $logger,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
    $this->eventDispatcher = $event_dispatcher;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('event_dispatcher'),
      $container->get('logger.channel.at_ls'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function process(Job $job): JobResult {
    $data = $job->getPayload();
    $translation_request_storage = $this->entityTypeManager->getStorage('at_ls_translation_request');
    /** @var \Drupal\at_ls\Entity\AtlsTranslationRequestInterface $translation_request */
    $translation_request = $translation_request_storage->load($data['id']);

    // Checks if the translation request was processed manually.
    if (!$translation_request->isProcessable()) {
      $message = $this->t('The translation request ID @id was already processed manually.', [
        '@id' => $data['id'],
      ]);
      $this->logger->warning($message);
      return JobResult::failure($message);
    }

    try {
      $this->eventDispatcher->dispatch(
        new AtlsTranslationRequestEvent($translation_request),
        AtlsTranslationRequestEvent::PROCESS
      );
      $message = $this->t('The translation request ID @id has been processed successfully.', ['@id' => $translation_request->id()]);
      $this->logger->info($message);
      return JobResult::success($message);
    }
    catch (\Exception $e) {
      $message = $this->t('The translation request ID @id could not be processed at this moment: @message', [
        '@id' => $translation_request->id(),
        '@message' => $e->getMessage(),
      ]);
      $this->logger->warning($message);
      if ($translation_request->get('translation_type')->value === AtlsStringInterface::ASYNC) {
        // Set number of retries to 42 and retry delay to 4 hours.
        // This will allow to try to process the translation request 1 week
        // every 4 hours.
        // @todo Add a configuration form to allow the admin to set the number
        // of retries and the retry delay.
        return JobResult::failure($message, 42, 14400);
      }
      else {
        return JobResult::failure($message);
      }
    }
  }

}
