<?php

namespace Drupal\at_ls\Service;

use Drupal\at_ls\AtlsTrait;
use Drupal\at_ls\Http\HttpClientInterface;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\key\KeyRepositoryInterface;

/**
 * Defines the AT-LS API Manager.
 */
class AtlsApiManager implements AtlsApiManagerInterface {

  use AtlsTrait;
  use StringTranslationTrait;

  /**
   * AtlsApiManager constructor.
   *
   * @param \Drupal\at_ls\Http\HttpClientInterface $client
   *   The HTTP client.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The logger channel.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory.
   * @param \Drupal\key\KeyRepositoryInterface $keyRepository
   *   The key repository service.
   * @param \Drupal\Core\Language\LanguageManagerInterface $languageManager
   *   The language manager.
   */
  public function __construct(
    protected HttpClientInterface $client,
    protected LoggerChannelInterface $logger,
    protected ConfigFactoryInterface $configFactory,
    protected KeyRepositoryInterface $keyRepository,
    protected LanguageManagerInterface $languageManager
  ) {}

  /**
   * {@inheritdoc}
   */
  public function deleteFileByToken(string $token, $source_language): ?array {
    return $this->request(
      'POST',
      'delFileByToken',
      [
        'form_params' => [
          'token' => $token,
        ],
      ],
      $source_language
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFileByToken(string $token, string $source_language): ?array {
    return $this->request(
      'GET',
      'getFileByToken',
      [
        'query' => [
          'token' => $token,
        ],
      ],
      $source_language
    );
  }

  /**
   * {@inheritdoc}
   */
  public function ping(): ?array {
    return $this->request(
      'GET',
      'ping'
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getList(
    string $source_language,
    string $target_language,
    DrupalDateTime $datetime = NULL
  ): ?array {
    if (!$datetime) {
      $datetime = new DrupalDateTime();
    }

    return $this->request(
      'GET',
      'getlist',
      [
        'query' => [
          'sourcelang' => $this->getAtlsLangCode($source_language),
          'targetlang' => $this->getAtlsLangCode($target_language),
          'datecutoff' => $datetime->format(static::DATETIME_FORMAT),
        ],
      ],
      $source_language
    );
  }

  /**
   * {@inheritdoc}
   */
  public function translateAsynchronous(
    string $source_language,
    string $target_language,
    string $filename,
    string $text,
    string $callbackurl = NULL,
    string $encoding = NULL,
  ): ?array {
    $config = $this->configFactory
      ->getEditable('at_ls.settings');
    return $this->request(
      'POST',
      'translateAsynchronous',
      [
        'form_params' => [
          'sourcelang' => $this->getAtlsLangCode($source_language),
          'targetlang' => $this->getAtlsLangCode($target_language),
          'filename' => $filename,
          'base64' => base64_encode($text),
          'callbackurl' => ($callbackurl) ?: '',
          'errnotifiersendto' => $config->get('notification_email') ?: '',
          'encoding' => ($encoding) ?: static::DEFAULT_ENCODING,
        ],
      ],
      $source_language
    );
  }

  /**
   * {@inheritdoc}
   */
  public function translateSynchronous(
    string $source_language,
    string $target_language,
    string $filename,
    string $text,
    string $encoding = NULL
  ): ?array {
    return $this->request(
      'POST',
      'translateSynchronous',
      [
        'form_params' => [
          'sourcelang' => $this->getAtlsLangCode($source_language),
          'targetlang' => $this->getAtlsLangCode($target_language),
          'filename' => $filename,
          'base64' => base64_encode($text),
          'encoding' => ($encoding) ?: static::DEFAULT_ENCODING,
        ],
      ],
      $source_language
    );
  }

  /**
   * Gets the API key.
   *
   * @param string $source_language
   *   The source language.
   *
   * @return string|null
   *   The API key.
   */
  private function getApiKey(string $source_language = NULL): ?string {
    $key_id = '';
    $api_keys = $this->configFactory
      ->getEditable('at_ls.settings')
      ->get('api_keys');

    if ($source_language) {
      foreach ($api_keys as $api_key) {
        if ($api_key['source_language'] === $source_language) {
          $key_id = $api_key['api_key'];
        }
      }
    }
    else {
      // If no source language is provided, use the first key as default.
      $key_id = $api_keys[0]['api_key'] ?? '';
    }

    if (empty($key_id)) {
      $this->logger->error($this->t('The AT-LS API key is not configured'));
      return NULL;
    }

    $key = $this->keyRepository->getKey($key_id);

    if (!$key) {
      $this->logger->error($this->t('The AT-LS API key is not found'));
      return NULL;
    }

    return $key->getKeyValue();
  }

  /**
   * Sends the API request.
   *
   * @param string $method
   *   The REST API method.
   * @param string $path
   *   The path of the API call.
   * @param array $options
   *   The request options.
   * @param string $source_language
   *   The source language.
   *
   * @return array|null
   *   The request response.
   */
  private function request(string $method, string $path, array $options = [], $source_language = NULL): ?array {
    $options['headers'] = [
      'X-ATRTS-API-Key' => $this->getApiKey($source_language),
      'Cache-Control' => 'no-cache',
    ];
    $options['timeout'] = static::REQUEST_TIMEOUT;

    $response = $this->client->handleRequest(
      $method,
      static::ENDPOINT . $path,
      $options
    );

    return Json::decode($response);
  }

}
