<?php

namespace Drupal\at_ls\Service;

use Drupal\Core\Datetime\DrupalDateTime;

/**
 * Provides an interface defining the AT-LS API manager service.
 */
interface AtlsApiManagerInterface {

  /**
   * The default datetime format.
   *
   * @var string
   */
  const DATETIME_FORMAT = 'Y-m-d\TH:i:s\Z';

  /**
   * The default encoding.
   *
   * @var string
   */
  const DEFAULT_ENCODING = 'UTF-8';

  /**
   * The API endpoint.
   *
   * @var string
   */
  const ENDPOINT = 'https://rts.at-ls.com/atrts/restapi/';

  /**
   * The request timeout value in seconds.
   *
   * @var int
   */
  const REQUEST_TIMEOUT = 10;

  /**
   * API call for delete file by token.
   *
   * @param string $token
   *   The petition token.
   * @param string $source_language
   *   The source language.
   *
   * @return array|null
   *   The delete file by token response
   */
  public function deleteFileByToken(string $token, string $source_language): ?array;

  /**
   * API call for get file by token.
   *
   * @param string $token
   *   The petition token.
   * @param string $source_language
   *   The source language.
   *
   * @return array|null
   *   The get file by token response
   */
  public function getFileByToken(string $token, string $source_language): ?array;

  /**
   * API call for ping.
   *
   * @return array|null
   *   The ping response
   */
  public function ping(): ?array;

  /**
   * API call for get list.
   *
   * @param string $source_language
   *   The source language.
   * @param string $target_language
   *   The target language.
   * @param \Drupal\Core\Datetime\DrupalDateTime $datetime
   *   The cut off date.
   *
   * @return array|null
   *   The get list response
   */
  public function getList(
    string $source_language,
    string $target_language,
    DrupalDateTime $datetime = NULL
  ): ?array;

  /**
   * API call for asynchronous translation.
   *
   * @todo Use the optional parameters callbackurl and errnotifiersendto.
   *
   * @param string $source_language
   *   The source language.
   * @param string $target_language
   *   The target language.
   * @param string $filename
   *   The filename.
   * @param string $text
   *   The text to translate.
   * @param string $callbackurl
   *   The callback url.
   * @param string $encoding
   *   The encoding.
   *
   * @return array|null
   *   The asynchronous translation response
   */
  public function translateAsynchronous(
    string $source_language,
    string $target_language,
    string $filename,
    string $text,
    string $callbackurl = NULL,
    string $encoding = NULL
  ): ?array;

  /**
   * API call for synchronous translation.
   *
   * @param string $source_language
   *   The source language.
   * @param string $target_language
   *   The target language.
   * @param string $filename
   *   The filename.
   * @param string $text
   *   The text to translate.
   * @param string $encoding
   *   The encoding.
   *
   * @return array|null
   *   The synchronous translation response
   */
  public function translateSynchronous(
    string $source_language,
    string $target_language,
    string $filename,
    string $text,
    string $encoding = NULL
  ): ?array;

}
