<?php

declare(strict_types=1);

namespace Drupal\atd;

use Drupal\Core\Extension\Extension;

/**
 * Manipulates profile, module and theme's meta information.
 *
 * @internal
 */
class AtdTranslationInfo {

  /**
   * The directory that contains the translation files.
   */
  private const TRANSLATION_DIRECTORY = 'translations';

  /**
   * Pattern to locate the .po file containing the translations.
   */
  private const INTERFACE_TRANSLATION_SERVER_PATTERN = '%project.%language.po';

  /**
   * Alter the information parsed from module and theme .info.yml files.
   *
   * This hook is invoked in \Drupal\Core\Extension\ExtensionList::doList(). A
   * module may implement this hook in order to add to or alter the data
   * generated by reading the .info.yml file with
   * \Drupal\Core\Extension\InfoParser.
   *
   * Using implementations of this hook to make modules required by setting the
   * $info['required'] key is discouraged. Doing so will slow down the module
   * installation and uninstallation process. Instead, use
   * \Drupal\Core\Extension\ModuleUninstallValidatorInterface.
   *
   * @param array $info
   *   The .info.yml file contents, passed by reference so that it can be
   *   altered.
   * @param \Drupal\Core\Extension\Extension $file
   *   Full information about the module or theme.
   * @param string $type
   *   Either 'module' or 'theme', depending on the type of .info.yml file that
   *   was passed.
   *
   * @see \Drupal\Core\Extension\ModuleUninstallValidatorInterface
   */
  public function systemInfoAlter(array &$info, Extension $file, string $type): void {
    // For all extensions that have a 'translations/' directory, add the
    // appropriate translation properties (so that the .po file containing the
    // translations is found).
    if (is_dir($file->getPath() . DIRECTORY_SEPARATOR . self::TRANSLATION_DIRECTORY)) {
      $info['interface translation project'] = $file->getName();
      $info['interface translation server pattern'] = $file->getPath() . DIRECTORY_SEPARATOR . self::TRANSLATION_DIRECTORY . DIRECTORY_SEPARATOR . self::INTERFACE_TRANSLATION_SERVER_PATTERN;
    }
  }

}
