<?php

/**
 * @file
 * Various non-administration page callbacks for the atom module.
 */

/**
 * Produces an atom 1.0 feed for the front page content.
 */
function atom_feed() {
  $nodes = db_query_range(db_rewrite_sql("SELECT n.nid FROM {node} n WHERE n.promote = '1' AND n.status = '1' ORDER BY n.created DESC"), 0, variable_get('atom_feed_entries', 15));

  $feed_info = array();
  $feed_info['html_url'] = url('', array('absolute' => TRUE));
  $feed_info['atom_url'] = url('atom/feed', array('absolute' => TRUE));
  _atom_print_feed($nodes, $feed_info);
}

function atom_blog_feed() {
  $nodes = db_query_range(db_rewrite_sql("SELECT n.nid FROM {node} n WHERE n.type = 'blog' AND n.status = '1' ORDER BY n.created DESC"), 0, variable_get('atom_feed_entries', 15));

  $feed_info = array();
  $feed_info['title']    = t('!site_name blogs', array('!site_name' => variable_get('site_name', 'Drupal')));
  $feed_info['html_url'] = url('blog', array('absolute' => TRUE));
  $feed_info['atom_url'] = url('blog/atom/feed', array('absolute' => TRUE));
  _atom_print_feed($nodes, $feed_info);
}

function atom_node_feed($node) {
  if (!node_access('view', $node)) {
    return drupal_access_denied();
  }

  $nodes = array($node->nid);

  $feed_info = array();
  $feed_info['html_url'] = url('node/' . $node->nid, array('absolute' => TRUE));
  $feed_info['atom_url'] = url('node/' . $node->nid . '/atom.xml', array('absolute' => TRUE));
  _atom_print_feed($nodes, $feed_info);
}

function atom_user_blog_feed($account) {
  $nodes = db_query_range(db_rewrite_sql("SELECT n.nid FROM {node} n WHERE n.type = 'blog' AND n.uid = %d AND n.status = '1' ORDER BY n.created DESC"), $account->uid, 0, variable_get('atom_feed_entries', 15));

  $feed_info = array();
  $feed_info['title']    = t("!name's blog", array('!name' => $account->name));
  $feed_info['subtitle'] = '';
  $feed_info['html_url'] = url('blog/' . $account->uid, array('absolute' => TRUE));
  $feed_info['atom_url'] = url('blog/' . $account->uid . '/atom/feed', array('absolute' => TRUE));
  _atom_print_feed($nodes, $feed_info);
}

function atom_taxonomy_feed($str_tids) {
  $terms = taxonomy_terms_parse_string($str_tids);
  $result = db_query(db_rewrite_sql('SELECT t.tid FROM {term_data} t WHERE t.tid IN ('. db_placeholders($terms['tids']) .')', 't', 'tid'), $terms['tids']);
  $tids = array();
  while ($tid = db_result($result)) {
    $tids[] = $tid;
  }
  $nodes = taxonomy_select_nodes($tids, $terms['operator'], 0, TRUE);

  $feed_info = array();
  if (count($tids) == 1) {
    $term = taxonomy_get_term($tids[0]);
    $feed_info['title']    = $term->name;
    $feed_info['subtitle'] = $term->description;
  }
  $feed_info['html_url'] = url("taxonomy/term/$str_tids", array('absolute' => TRUE));
  $feed_info['atom_url'] = url("taxonomy/term/$str_tids/atom/feed", array('absolute' => TRUE));
  _atom_print_feed($nodes, $feed_info);
}

/**
 * prints feed information from query of either front page or blog content
 *
 * @param object $nodes query result
 * @param array feed information
 */
function _atom_print_feed($nodes, $feed_info) {
  $output = '';
  $last_mod = 0;
  while ($node = db_fetch_object($nodes)) {
    $node = node_load(array('nid' => $node->nid));
    $node->build_mode = NODE_BUILD_RSS;
    $node->link = url("node/$node->nid", array('absolute' => TRUE));

    if (node_hook($node, 'view')) {
      node_invoke($node, 'view', TRUE, FALSE);
    }
    else {
      $node = node_prepare($node, TRUE);
    }

    // Allow modules to change $node->teaser before viewing.
    node_invoke_nodeapi($node, 'rss item', TRUE, FALSE);
    node_invoke_nodeapi($node, 'alter', TRUE);

    $item = array(
      'published' => $node->created,
      'updated' => $node->changed,
      'author' => $node->name ? $node->name : variable_get('anonymous', 'Anonymous'),
      'summary' => variable_get('atom_display_content', TRUE) ? check_markup($node->body, $node->format, FALSE) : '',
      'content' => variable_get('atom_display_summary', TRUE) ? check_markup($node->teaser, $node->format, FALSE) : '',
    );

    if (module_exists('taxonomy')) {
      $terms = taxonomy_node_get_terms($node);
      foreach ($terms as $term) {
        $extra[] = array('key' => 'category', 'attributes' => array('term' => $term->name), 'value' => '');
      }
    }

    $output .= theme('atom_feed_item', $node->title, $node->link, $item, $extra);
  }

  // Merge some default values.
  $feed_info += array(
    'title' => variable_get('site_name', 'Drupal'),
    'subtitle' => variable_get('site_slogan', ''),
  );

  $output = theme('atom_feed', $feed_info, $output);

  drupal_set_header('Content-Type: application/atom+xml; charset=utf-8');
  print $output;
}

function theme_atom_feed(array $feed_info, $items) {
  $output = "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n";
  $output .= '<feed xmlns="http://www.w3.org/2005/Atom">'."\n";
  $output .= '  <title type="text">'. check_plain($feed_info['title']) ."</title>\n";
  if ($feed_info['subtitle']) {
    $output .= '  <subtitle type="text">' . check_plain($feed_info['subtitle']) . "</subtitle>\n";
  }
  $output .= '  <link rel="alternate" type="text/html" href="'. check_plain($feed_info['html_url']) .'" />'."\n";
  $output .= '  <link rel="self" type="application/atom+xml" href="'. check_plain($feed_info['atom_url']) .'" />'."\n";
  //$output .= '  <generator version="' . VERSION . '" uri="http://drupal.org">Drupal</generator>'."\n";
  $output .= '  <id>'. check_plain($feed_info['atom_url']) ."</id>\n";
  $output .= '  <updated>'. gmdate(DATE_ATOM, time()) ."</updated>\n";
  $output .= $items;
  $output .= "</feed>\n";
  return $output;
}

function theme_atom_feed_item($title, $link, $item, $extra = array()) {
  $item += array(
    'id' => $link,
    'summary' => '',
    'content' => '',
    'author' => '',
  );

  $output = "  <entry>\n";
  $output .= "    <id>" . check_plain($item['id']) . "</id>\n";
  $output .= "    <link rel=\"alternate\" type=\"text/html\" href=\"" . check_plain($link) . "\" />\n";
  $output .= "    <published>" . gmdate(DATE_ATOM, $item['published']) . "</published>\n";
  $output .= "    <updated>" . gmdate(DATE_ATOM, $item['updated']) . "</updated>\n";
  $output .= "    <title type=\"text\">" . check_plain($title) . "</title>\n";
  if ($item['author']) {
    $output .= "    <author><name>" . check_plain($item['author']) . "</name></author>\n";
  }
  if ($item['summary']) {
    $output .= "    <summary type=\"xhtml\"><div xmlns=\"http://www.w3.org/1999/xhtml\">" . $item['summary'] . "</div></summary>\n";
  }
  if ($item['content']) {
    $output .= "    <content type=\"xhtml\"><div xmlns=\"http://www.w3.org/1999/xhtml\">" . $item['content'] . "</div></content>\n";
  }
  if ($extra) {
    $output .= format_xml_elements($extra);
  }
  $output .= "  </entry>\n";
  return $output;
}
