<?php

/**
 * @file
 * common.inc
 */

use drupol\htmltag\Attributes\AttributesFactory;
use drupol\htmltag\Attribute\AttributeFactory;
use Drupal\atomium\AtomiumClassAttribute;
use Drupal\atomium\AtomiumPlaceholderAttribute;

/**
 * Includes a theme file.
 *
 * @param string $theme
 *   Name of the theme to use for base path.
 * @param string $path
 *   Path relative to $theme.
 *
 * @return bool
 *   TRUE if file has been included, FALSE otherwise.
 */
function atomium_include($theme, $path) {
  static $themes = array();

  $themes += array(
    $theme => drupal_get_path('theme', $theme),
  );

  if ($themes[$theme] && ($file = DRUPAL_ROOT . '/' . $themes[$theme] . '/' . $path) && \file_exists($file)) {
    if (\is_dir($file)) {
      foreach (\glob($file . '/*.{inc,php}', GLOB_BRACE) as $file_to_include) {
        include_once $file_to_include;
      }
    }

    if (\is_file($file)) {
      include_once $file;
    }

    return TRUE;
  }

  return FALSE;
}

/**
 * Returns a list of base themes for active or provided theme.
 *
 * @param string $theme_key
 *   The machine name of the theme to check, if not set the active theme name
 *   will be used.
 * @param bool $include_theme_key
 *   Whether to append the returned list with $theme_key.
 *
 * @return array
 *   An indexed array of base themes.
 */
function _atomium_get_base_themes($theme_key = NULL, $include_theme_key = FALSE) {
  static $themes;

  $theme_key = $theme_key ?: $GLOBALS['theme_key'];

  if (!isset($themes[$theme_key])) {
    $themes[$theme_key] = \array_unique(
      \array_filter(
        (array) atomium_get_theme_info($theme_key, 'base theme')
      )
    );
  }

  if ($include_theme_key) {
    $themes[$theme_key][] = $theme_key;
  }

  return \array_unique($themes[$theme_key]);
}

/**
 * Return information from the .info file of a theme (and possible base themes).
 *
 * @param string $theme_key
 *   The machine name of the theme.
 * @param string $key
 *   The key name of the item to return from the .info file. This value can
 *   include "][" to automatically attempt to traverse any arrays.
 * @param bool $base_themes
 *   Recursively search base themes, defaults to TRUE.
 *
 * @return string|array|false
 *   A string or array depending on the type of value and if a base theme also
 *   contains the same $key, FALSE if no $key is found.
 *
 * @SuppressWarnings(PHPMD.CyclomaticComplexity)
 * @SuppressWarnings(PHPMD.NPathComplexity)
 */
function atomium_get_theme_info($theme_key = NULL, $key = NULL, $base_themes = TRUE) {
  // If no $theme_key is given, use the current theme if we can determine it.
  if ($theme_key === NULL) {
    $theme_key = !empty($GLOBALS['theme_key']) ? $GLOBALS['theme_key'] : NULL;
  }

  if ($theme_key === NULL) {
    return FALSE;
  }

  $themes = list_themes();
  if (empty($themes[$theme_key])) {
    return FALSE;
  }

  $theme = $themes[$theme_key];

  if ($key === NULL) {
    return $theme->info;
  }

  $value = FALSE;

  // Recursively add base theme values.
  if ($base_themes === TRUE && isset($theme->base_themes)) {
    foreach (\array_keys($theme->base_themes) as $base_theme) {
      $value = atomium_get_theme_info($base_theme, $key);
    }
  }

  $info = $themes[$theme_key]->info;
  // Allow array traversal.
  foreach (\explode('][', $key) as $parent) {
    if (isset($info[$parent])) {
      $info = (array) $info[$parent];
    }
    else {
      $info = array();
    }
  }

  if (!empty($info)) {
    if (\is_array($value)) {
      $value = drupal_array_merge_deep($value, $info);
    }
    elseif (empty($value)) {
      $value = $info;
    }
    else {
      $value = drupal_array_merge_deep((array) $value, $info);
    }
  }

  return $value;
}

/**
 * Retrieves an element's "attributes" array.
 *
 * @param array $element
 *   The individual renderable array element. It is possible to also pass the
 *   $variables parameter in [pre]process functions and it will logically
 *   determine the correct path to that particular theme hook's attribute array.
 *   Passed by reference.
 * @param string $property
 *   Determines which attributes array to retrieve. By default, this is the
 *   normal attributes, but can be "wrapper_attributes" or
 *   "input_group_attributes".
 *
 * @return array
 *   The attributes array. Passed by reference.
 */
function &_atomium_get_attributes(array &$element, $property = 'attributes') {
  $hash_property = '#' . $property;

  // Attempt to retrieve a renderable element attributes first.
  if (
    isset($element['#type']) ||
    isset($element['#theme']) ||
    isset($element['#pre_render']) ||
    isset($element['#markup']) ||
    isset($element['#theme_wrappers']) ||
    isset($element[$hash_property])
  ) {
    $element += array(
      $hash_property => array(),
    );

    return $element[$hash_property];
  }

  // Treat $element as if it were a [pre]process function $variables parameter
  // and look for a renderable "element".
  if (isset($element['element'])) {
    $element['element'] += array(
      $hash_property => array(),
    );

    return $element['element'][$hash_property];
  }

  // If all else fails, create (if needed) a default "attributes" array. This
  // will, at the very least, either work or cause an error that can be tracked.
  $element += array(
    $property => array(),
  );

  return $element[$property];
}

/**
 * Discover components across a theme and all its subthemes.
 */
function atomium_find_templates() {
  foreach (_atomium_get_base_themes(NULL, TRUE) as $theme) {
    $path = drupal_get_path('theme', $theme) . '/templates';

    if (!\is_dir($path)) {
      continue;
    }

    $directories = \array_diff(
      \scandir($path, SCANDIR_SORT_ASCENDING),
      array('..', '.')
    );

    foreach ($directories as $directory) {
      $component_directory = $path . DIRECTORY_SEPARATOR . $directory;

      if (!\is_dir($component_directory)) {
        continue;
      }

      yield array(
        'theme' => $theme,
        'component' => $directory,
        'directory' => $component_directory,
        'includes' => \glob($component_directory . '/*.component.inc'),
      );
    }
  }
}

/**
 * Verbatim copy of drupal_get_breadcrumb().
 *
 * The only modified stuff are the call to menu_get_active_breadcrumb() which
 * is replaced with atomium_menu_get_active_breadcrumb().
 *
 * @see drupal_get_breadcrumb()
 */
function atomium_drupal_get_breadcrumb() {
  $breadcrumb = drupal_set_breadcrumb();

  if (!isset($breadcrumb)) {
    $breadcrumb = atomium_menu_get_active_breadcrumb();
  }

  return $breadcrumb;
}

/**
 * Verbatim copy of menu_get_active_breadcrumb().
 *
 * The difference with the original function is that this function returns
 * a render array instead of an array of links rendered.
 *
 * This will allow us to modify the data properly.
 *
 * @see menu_get_active_breadcrumb()
 */
function atomium_menu_get_active_breadcrumb() {
  $breadcrumb = array();

  // No breadcrumb for the front page.
  if (drupal_is_front_page()) {
    return $breadcrumb;
  }

  $item = menu_get_item();
  if (!empty($item['access'])) {
    $active_trail = menu_get_active_trail();

    // Allow modules to alter the breadcrumb, if possible, as that is much
    // faster than rebuilding an entirely new active trail.
    drupal_alter('menu_breadcrumb', $active_trail, $item);

    // Don't show a link to the current page in the breadcrumb trail.
    $end = \end($active_trail);
    if ($item['href'] === $end['href']) {
      \array_pop($active_trail);
    }

    // Remove the tab root (parent) if the current path links to its parent.
    // Normally, the tab root link is included in the breadcrumb, as soon as we
    // are on a local task or any other child link. However, if we are on a
    // default local task (e.g., node/%/view), then we do not want the tab root
    // link (e.g., node/%) to appear, as it would be identical to the current
    // page. Since this behavior also needs to work recursively (i.e., on
    // default local tasks of default local tasks), and since the last non-task
    // link in the trail is used as page title (see menu_get_active_title()),
    // this condition cannot be cleanly integrated into menu_get_active_trail().
    // menu_get_active_trail() already skips all links that link to their parent
    // (commonly MENU_DEFAULT_LOCAL_TASK). In order to also hide the parent link
    // itself, we always remove the last link in the trail, if the current
    // router item links to its parent.
    if (($item['type'] & MENU_LINKS_TO_PARENT) === MENU_LINKS_TO_PARENT) {
      \array_pop($active_trail);
    }

    foreach ($active_trail as $parent) {
      $breadcrumb_part = array(
        '#theme' => 'link',
        '#text' => $parent['title'],
        '#path' => $parent['href'],
        '#options' => $parent['localized_options'],
      );
      _atomium_extend_theme_property(
        $breadcrumb_part,
        array('breadcrumb'));
      $breadcrumb[] = $breadcrumb_part;
    }
  }

  return $breadcrumb;
}

/**
 * Custom utility callback that apply to each array element, a callback.
 *
 * Unlike array_walk_recursive, this function send the key name if a value is
 * an array.
 * As the value is passed by reference, you are able to modify its value within
 * the callback.
 *
 * @param array $variables
 *   The array.
 * @param callable $callback
 *   The callback function.
 */
function atomium_recursive_foreach(array &$variables, callable $callback) {
  $variables['#atomium_recursive_processed'] = TRUE;

  // By default set the recursive to FALSE.
  $variables += array(
    '#atomium_recursive' => FALSE,
  );

  foreach ($variables as $key => &$value) {
    if (\is_array($value) && !isset($value['#atomium_recursive_processed'])) {

      // If no recursive is set, take the value by default.
      $value += array(
        '#atomium_recursive' => $variables['#atomium_recursive'],
      );

      if ($value['#atomium_recursive'] === TRUE) {
        atomium_recursive_foreach($value, $callback);
      }

      // Remove the key to avoid polluting the array with unwanted values.
      unset($value['#atomium_recursive']);
    }

    \call_user_func_array($callback, array($key, &$value, &$variables));
  }

  unset($variables['#atomium_recursive_processed']);
}

/**
 * Flatten an array.
 *
 * If input array is:
 *
 * $input = array(
 *   'parent1' => 'valueA'
 *   'parent2' => array(
 *     'subparent2' => 'valueB',
 *   'parent3' => array(
 *     'subparent3' => array(
 *       'valueC1',
 *       'valueC2',
 *       'valueC3',
 *       'valueC4',
 *     )
 *   )
 * );
 *
 * The output array $storage will be:
 *
 * array(
 *   'parent1' => 'valueA',
 *   'parent2.subparent2' => 'valueB',
 *   'parent3.subparent3' => array(
 *       'valueC1',
 *       'valueC2',
 *       'valueC3',
 *       'valueC4',
 *   ),
 * );
 *
 * @param array $array
 *   The input array.
 * @param array $storage
 *   The output array.
 * @param string $parentKey
 *   The parent key where to start from.
 */
function _atomium_array_flatten(array $array, array &$storage, $parentKey = '') {
  foreach ($array as $key => $value) {
    $itemKey = $parentKey;

    if (!\is_numeric($key)) {
      $itemKey .= ($parentKey ? '.' : '') . $key;
    }

    if (\is_array($value)) {
      _atomium_array_flatten($value, $storage, $itemKey);
    }
    else {
      if (\is_numeric($key)) {
        $storage += array($itemKey => array());
        $storage[$itemKey][] = $value;
      }
      else {
        $storage[$itemKey] = $value;
      }
    }
  }
}

/**
 * Get settings from the current theme info file.
 *
 * @param string $setting_keys
 *   The settings to get, flattened (concatenated with ".")
 * @param bool $base_themes
 *   TRUE to get the value in the base_themes
 *   if the settings is not found in current theme,
 *   FALSE otherwise.
 * @param array $value_callbacks
 *   The list of callbacks to apply to retrieved values.
 *
 * @return array
 *   The settings as an array.
 */
function atomium_get_settings($setting_keys, $base_themes = TRUE, array $value_callbacks = array('trim')) {
  static $drupal_static_fast;

  if (!isset($drupal_static_fast)) {
    $drupal_static_fast['atomium_get_settings'] = &drupal_static(__FUNCTION__);
  }

  $theme_key = $GLOBALS['theme_key'];
  $cid = $theme_key;
  if ($base_themes) {
    $cid .= '+';
  }

  $settings_by_theme = &$drupal_static_fast['atomium_get_settings'];

  if (!isset($settings_by_theme[$cid])) {
    $settings_by_theme[$cid] = array();
    if ($theme_settings = atomium_get_theme_info($theme_key, 'settings', $base_themes)) {
      _atomium_array_flatten(
        $theme_settings,
        $settings_by_theme[$cid]);
    }
  }

  $settings = $settings_by_theme[$cid];

  $setting_value = array();

  if (isset($settings[$setting_keys])) {
    if (\is_string($settings[$setting_keys])) {
      $setting_value = \explode(',', $settings[$setting_keys]);
    }

    if (\is_array($settings[$setting_keys])) {
      $setting_value = $settings[$setting_keys];
    }

    foreach ($value_callbacks as $value_callback) {
      $setting_value = \array_map($value_callback, $setting_value);
    }
  }

  return \array_values(\array_filter($setting_value));
}

/**
 * Get the base hook of a hook name.
 *
 * Example:
 *   If input hook is form_element__checkbox__form_random_id,
 *   then the output base hook will be: form_element.
 *
 * @param string $hook
 *   The hook.
 *
 * @return string
 *   The base hook.
 */
function atomium_get_base_hook($hook) {
  if (FALSE !== $pos = \strpos($hook, '__')) {
    return \substr($hook, 0, $pos);
  }

  return $hook;
}

/**
 * Retrieves output to be displayed in the HEAD tag of the HTML page.
 *
 * @param array $context
 *   The context.
 *
 * @return array
 *   The element.
 */
function atomium_drupal_get_html_head(array $context = array()) {
  $elements = drupal_add_html_head();

  if (!module_exists('registryonsteroids_alter')) {
    watchdog(
      'Atomium theme',
      'The Registry on steroids Alter module is not installed',
      array(),
      WATCHDOG_ERROR
    );

    return $elements;
  }

  registryonsteroids_alter_recursive_element_children(
    $elements,
    function (&$element, $element_key, &$context) {
      if (!isset($element['#tag'])) {
        return NULL;
      }
      _atomium_extend_theme_property(
        $element,
        array($element['#tag'], $context['theme_hook_original'])
      );
    },
    NULL,
    $context
  );

  return $elements;
}

/**
 * Wrapper for Registry on steroids Alter functions.
 *
 * @param array $element
 *   A Drupal element.
 * @param string[] $suggestions_parts
 *   The suggestions strings.
 */
function _atomium_extend_theme_property(array &$element, array $suggestions_parts) {
  if (!module_exists('registryonsteroids_alter')) {
    watchdog(
      'Atomium theme',
      'The Registry on steroids Alter module is not installed',
      array(),
      WATCHDOG_ERROR
    );
  }
  else {
    _registryonsteroids_alter_extend_theme_property_with_suggestions(
      $element,
      $suggestions_parts
    );
  }
}

/**
 * Wrapper for Registry on steroids Alter functions.
 *
 * @param array $element
 *   A Drupal element.
 * @param array $suggestions_parts
 *   The suggestions strings.
 */
function _atomium_extend_theme_wrappers_property(array &$element, array $suggestions_parts) {
  if (!module_exists('registryonsteroids_alter')) {
    watchdog(
      'Atomium theme',
      'The Registry on steroids Alter module is not installed',
      array(),
      WATCHDOG_ERROR
    );
  }
  else {
    _registryonsteroids_alter_extend_theme_wrappers_property_with_suggestions(
      $element,
      $suggestions_parts
    );
  }
}

/**
 * Wrapper for Registry on steroids Alter functions.
 *
 * @param array|string $hook
 *   The Drupal theme hook.
 * @param array $suggestions_parts
 *   The suggestions strings.
 */
function _atomium_extend_value_with_suggestions(&$hook, array $suggestions_parts) {
  if (!module_exists('registryonsteroids_alter')) {
    watchdog(
      'Atomium theme',
      'The Registry on steroids Alter module is not installed',
      array(),
      WATCHDOG_ERROR
    );
  }
  else {
    _registryonsteroids_alter_extend_value_with_suggestions(
      $hook,
      $suggestions_parts
    );
  }
}

/**
 * Get an instance of an Atomium attributes object.
 *
 * @param array $values
 *   The initial values of the attributes.
 *
 * @return \drupol\htmltag\Attributes\AttributesInterface
 *   An attributes object.
 */
function atomium_get_attributes(array $values) {
  AttributeFactory::$registry['class'] = AtomiumClassAttribute::class;
  AttributeFactory::$registry['placeholder'] = AtomiumPlaceholderAttribute::class;

  return AttributesFactory::build($values);
}
