<?php

namespace Drupal\attachinbody;

use Drupal\Core\Asset\AssetResolverInterface;
use Drupal\Core\Asset\AttachedAssetsInterface;
use Drupal\Core\Asset\LibraryDiscoveryInterface;

/**
 * Decorate the core asset resolver to remove in-body libraries.
 */
class AssetResolverDecorator implements AssetResolverInterface {

  /**
   * The Asset Resolver Service.
   *
   * @var \Drupal\Core\Asset\AssetResolverInterface
   */
  private $decorated;

  /**
   * The Library Discovery service.
   *
   * @var \Drupal\Core\Asset\LibraryDiscoveryInterface
   */
  private $libraryDiscovery;

  /**
   * The storage for libraries included in the page body.
   *
   * @var \Drupal\attachinbody\InBodyLibraries
   */
  private $inBodyLibraries;


  /**
   * AssetResolverDecorator constructor.
   *
   * @param \Drupal\Core\Asset\AssetResolverInterface $decorated
   *   The Asset Resolver service.
   * @param \Drupal\attachinbody\TwigExtension $twigExtension
   *   The AttachInBody TwigExtension.
   */
  public function __construct(AssetResolverInterface $decorated, LibraryDiscoveryInterface $libraryDiscovery, InBodyLibraries $inBodyLibraries) {
    $this->decorated = $decorated;
    $this->libraryDiscovery = $libraryDiscovery;
    $this->inBodyLibraries = $inBodyLibraries;
  }

  /**
   * {@inheritdoc}
   */
  public function getCssAssets(AttachedAssetsInterface $assets, $optimize) {
    $this->updateAssets($assets);

    return $this->decorated->getCssAssets($assets, $optimize);
  }

  /**
   * {@inheritdoc}
   */
  public function getJsAssets(AttachedAssetsInterface $assets, $optimize) {
    $this->updateAssets($assets);

    return $this->decorated->getJsAssets($assets, $optimize);
  }

  /**
   * Update the AttachedAssets for libraries included in the page body.
   *
   * @param \Drupal\Core\Asset\AttachedAssetsInterface $assets
   *   The attached assets for the page.
   */
  private function updateAssets(AttachedAssetsInterface $assets) {
    // Set a property on the asset object so that it's only processed once.
    if ($assets->_updatedInBodyAssets ?? FALSE) {
      return;
    }
    $assets->_updatedInBodyAssets = TRUE;

    // Replace the in-body library with its dependencies.
    $libraries = $assets->getLibraries();
    foreach ($this->inBodyLibraries->getLibraries() as $library) {
      list($extension, $name) = explode('/', $library, 2);
      $definition = $this->libraryDiscovery->getLibraryByName($extension, $name);
      $libraries = array_merge($libraries, $definition['dependencies'] ?? []);

      $libraryIndex = array_search($library, $libraries);
      unset($libraries[$libraryIndex]);
    }
    $assets->setLibraries($libraries);
  }

}
