<?php

declare(strict_types=1);

namespace Drupal\attempt_mgmt;

use Drupal\Core\Database\Connection;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\Session\AccountInterface;
use Drupal\Component\Utility\Crypt;


/**
 * @todo Add class description.
 */
final class AttemptFactory {

  use LoggerChannelTrait;

  /**
   * Constructs an AttemptFactory object.
   */
  public function __construct(
    private readonly Connection $connection,
    private readonly AccountProxyInterface $currentUser,
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly EntityFieldManagerInterface $entityFieldManager,
  ) {}
  
  /**
   * @todo Add method description.
   */
  public function settingExists($data): bool {
    $query = $this->connection->select('attempt_mgmt_settings');
    $query->fields('attempt_mgmt_settings', ['entity_type', 'entity_id', 'settings']);
    $query->condition('entity_type', $data['entity_type'], '=')
      ->condition('entity_id', $data['entity_id'], '=')
      ->range(0, 1);

    $result = $query->execute();

    if (!empty($result->fetchAssoc())) {
      return TRUE;
    }

    return FALSE;
    
  }

  public function loadSettings($data) {

    $settings = [];

    $query = $this->connection->select('attempt_mgmt_settings');
    $query->fields('attempt_mgmt_settings', ['entity_type', 'entity_id', 'settings']);
    $query->condition('entity_type', $data['entity_type'], '=')
      ->condition('entity_id', $data['entity_id'], '=')
      ->range(0, 1);

    $result = $query->execute();
  
    if (!empty($result)) {
      $settings = $result->fetchField(2);
      $settings = unserialize($settings);   
    }

    return $settings;


  }

  /**
   * {@inheritdoc}
   */
  public function updateSettings($data) {

    return $this->connection
      ->update('attempt_mgmt_settings')
      ->fields($data)
      ->condition('entity_type', $data['entity_type'], '=')
      ->condition('entity_id', $data['entity_id'], '=')
      ->execute();
  }

  /**
   * {@inheritdoc}
   */
  public function insertSettings($data) {

    return $this->connection
      ->insert('attempt_mgmt_settings')
      ->fields($data)
      ->execute();
  }  
 
  /**
   * Number attempts per user
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity this attempt has been attached to.
   * @return int
   *   The number of attempts found for the current user.
   */
  public function numberAttemptsPerUser(EntityInterface $entity, $user_id, $session_id)  {
    $query = $this->entityTypeManager->getStorage('attempt_mgmt_attempt')->getQuery();

    // Search for session_id on anonymous users.
    if ($user_id == 0 && !empty($session_id)) {
      $query->condition('session_uuid', $session_id);
    }
    else {
      $query->condition('uid', $user_id);
    }

    // Do not count temporary attempts!
    $query->condition('temporary', FALSE);

    $query
      ->condition('entity_type', $entity->getEntityTypeId())
      ->condition('entity_id', $entity->id())
      ->accessCheck(FALSE);
    
    $number_attempts = $query->count()->execute();

    return $number_attempts;

  }

  /**
   * Get the best attempt.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity object.
   * @param string $sort_field
   *   The given sort field for the sort.
   * @return \Drupal\Core\Entity\EntityInterface
   *   The found record or NULL.
   */
  public function getBestAttempt(EntityInterface $entity, $sort_field = NULL) {

    $record = NULL;

    $storage = $this->entityTypeManager->getStorage('attempt_mgmt_attempt');

    if (isset($sort_field)) {
      $query = $storage->getQuery();
      $query->condition('uid', $this->currentUser->id())
        ->condition('entity_type', $entity->getEntityTypeId())
        ->condition('entity_id', $entity->id())
        ->sort($sort_field , 'DESC') 
        ->range(0,1)
        ->accessCheck(FALSE);

      $entity_ids = $query->execute();

      if (!empty($entity_ids)) {
        $attempts = $storage->loadMultiple($entity_ids);        
        if ($attempt = reset($attempts)) {
          $record = $attempt;
        }
      }
    }

    return $record;  


  }

  protected function getAvailableAttemptManagementField($entity_type, $bundle) {

    $field_name = NULL;

    $fields = $this->entityFieldManager
      ->getFieldDefinitions($entity_type, $bundle);

    // Check for a attempt settings field
    foreach ($fields as $key => $field) {
      if ($field->getType() === 'attempt_mgmt_attempt_settings') {  
        $field_name = $field->getName();
      }
    }

    return $field_name;  

  }  

  public function getAttemptFieldByProperty(EntityInterface $entity, $property) {

    $field_value = NULL; 

    if ($field = $this->getAttemptField($entity)) {
      $field_value = $field[$property];
    }

    return $field_value;

  }

  public function getAttemptField(EntityInterface $entity) {

    $field = NULL;

    if ($attempt_field = $this->getAvailableAttemptManagementField($entity->getEntityTypeId(), $entity->bundle())) {
      if ($field = $entity->get($attempt_field)->getValue()) {
        $field = $field[0];
      }
    }

    return $field;

  }
  
  /**
   * Creates and returns a unique identifier for unauthenticated users.
   * https://www.drupal.org/node/3006306
   *  
   * @return string $session_key
   *   Unique session key.
   */
  public function getSessionIDForUnauthenticatedUsers() {
    // Define session.
    $session = \Drupal::request()->getSession();
    // Check if we have already an unique identifier saved into session
    if (!$session->has('core.tempstore.private.owner')) {
     // This generates a unique identifier for the user
     $session->set('core.tempstore.private.owner', Crypt::randomBytesBase64());
    }
    
    $session_key = $session->get('core.tempstore.private.owner');
  
    return $session_key;
  
  } 

  public function checkExistingAttempt(EntityInterface $entity, $user_id) {

    $attempt_exists = FALSE;

    // For anon. user we need to save the session id.
    if ($user_id == 0) {
      $session_id = $this->getSessionIDForUnauthenticatedUsers();
    }
    else {
      $session_id = NULL;
    }
       
    // Get entity we attached attempt to
    $entity = $this->entityTypeManager->getStorage($entity_type_id)->load($entity_id);
    if ($entity instanceof EntityInterface) {
      if ($this->getExistingAttempt($entity, $account->id(), $session_id)) {
        $attempt_exists = TRUE;
      }
    }

    return $attempt_exists;

  }
  
  public function createAttempt(EntityInterface $entity, $user_id, $session_id = NULL) {

    $attempt_uuid = '';

    if ($session_id == NULL) {
      // For anon. user we need to save the session id.
      if ($user_id == 0) {
        $session_id = $this->getSessionIDForUnauthenticatedUsers();
      }
      else {
        $session_id = NULL;
      }
    }

    $number_attempts = $this->numberAttemptsPerUser($entity, $user_id, $session_id);

    if ($number_attempts > 0) {
      $update_previous_attempt = TRUE;
    }
    else {
      $update_previous_attempt = FALSE;
    }

    if ($entity) {

      // Write attempt and return UUID
      if ($attempt_type = $this->getAttemptFieldByProperty($entity, 'attempt_type')) {  

        try {
          $storage = $this->entityTypeManager->getStorage('attempt_mgmt_attempt');
          $attempt = $storage->create([
            'bundle' => $attempt_type,
            'entity_type' => $entity->getEntityTypeId(),
            'entity_id' => $entity->id(),
            'uid' => $user_id,
            'session_uuid' => $session_id,
            'number_attempt' => $number_attempts + 1,
            'temporary' => TRUE,  
            'status' => TRUE,
          ]);

          $attempt->save();
          $attempt_uuid = $attempt->uuid();

          if ($update_previous_attempt) {
            $this->updatePreviousAttempt($entity, $user_id, $session_id, $number_attempts);
          }

        }
        catch (\Exception $exception) {
          $error_message = $exception->getMessage();
          $this->getLogger('attempt_mgmt')->debug('Error when saving attempt. Please check the given fields. Info: @error_message.',['@error_message' => $error_message]);
        }
      
      }
    }

    return $attempt_uuid;

  }

  protected function updatePreviousAttempt(EntityInterface $entity, $user_id, $session_id, $number_attempt) {
    $storage = $this->entityTypeManager->getStorage('attempt_mgmt_attempt');
    $query = $storage->getQuery();
    $query->condition('entity_type', $entity->getEntityTypeId())
          ->condition('entity_id', $entity->id())
          ->condition('number_attempt', $number_attempt)
          ->condition('closed', FALSE)
          ->condition('temporary', FALSE)          
          ->accessCheck(FALSE);

    // Search for session_id on anonymous users.
    if ($user_id == 0 && !empty($session_id)) {
      $query->condition('session_uuid', $session_id);
    }
    else {
      $query->condition('uid', $user_id);
    }

    $results = $query->execute();

    if (!empty($results)) {
      $attempts = $storage->loadMultiple($results);
      if ($attempt = reset($attempts)) {
        $attempt->closed = TRUE;
        $attempt->save();
      }      
    }

  }
  
  /**
   * Check if attempt closed
   * 
   * @param string $attempt_uuid
   *   The uuid for the attempt
   * 
   * @return bool 
   *   TRUE | FALSE.  
   */
  protected function isAttemptClosed($attempt_uuid) {
    $attempts = $this->entityTypeManager->getStorage('attempt_mgmt_attempt')->loadByProperties(['uuid' => $attempt_uuid, 'closed' => TRUE]);
    if (!empty($attempts)) {
      return TRUE;
    }
    return FALSE;
  }

  public function updateAttempt(string $attempt_uuid, array $field_data, EntityInterface $entity, $user_id, $finished = FALSE): bool {

    $success = FALSE;

    // First let's check if the attempt is closed. When closed no update should be done!
    //$attempt_closed = $this->isAttemptClosed($attempt_uuid);
    //if ($attempt_closed) {
      //return $success;
    //}

    // For anon. user we need to save the session id.
    if ($user_id == 0) {
      $session_id = $this->getSessionIDForUnauthenticatedUsers();
    }
    else {
      $session_id = NULL;
    }

    $attempts = $this->entityTypeManager->getStorage('attempt_mgmt_attempt')->loadByProperties(['uuid' => $attempt_uuid]);

    if ($attempt = reset($attempts)) {
      // We need to set temporary to false.
      $attempt->temporary = FALSE;
      // In case the scorm has finished we need to close the attemt
      if ($finished) {
        $attempt->closed = TRUE;
      }
      // Field data
      foreach ($field_data as $field => $value) {
        $attempt->{$field} = $value;
      }
      $attempt->save();
      $success = TRUE;
    }

    return $success;

  }

  /**
   * Check if we need to force a new attempt
   *
   * @param EntityInterface $entity
   *   The entity we have attached attempt management.
   * @return bool
   *   TRUE | FALSE
   */
  public function isForceNewAttempt(EntityInterface $entity) {
    $force_new_attempt = FALSE;

    if ($this->getAttemptFieldByProperty($entity, 'force_new_attempt')) {
      $force_new_attempt = TRUE;
    }

    return $force_new_attempt;

  }

  /**
   * Get the entity.
   *
   * @param string $entity_type_id
   *   The entity_type_id.
   * @param string $entity_id
   *   The entity_id.
   * @return bool
   *   TRUE | FALSE
   */
  public function getEntity($entity_type_id, $entity_id) {
    $entity = $this->entityTypeManager->getStorage($entity_type_id)->load($entity_id);

    if ($entity instanceof EntityInterface) {
      return $entity;
    }

    return NULL;

  }  

  /**
   * Get the last attempt date for the given user or session.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity for that attempt.
   * @param int $user_id
   *   The user id for that attempt.
   * @param string $session_id
   *   The session id for that attempt.
   * @return 
   *   NULL | Timestamp for last attempt
   *   
   */
  protected function getLastAttemptDateForUser(EntityInterface $entity, $user_id, $session_id) {
    $last_attempt = NULL;
    $storage = $this->entityTypeManager->getStorage('attempt_mgmt_attempt');
    $query = $storage->getQuery();
    $query->condition('entity_type', $entity->getEntityTypeId())
          ->condition('entity_id', $entity->id())
          ->accessCheck(FALSE);

    // Search for session_id on anonymous users.
    if ($user_id == 0 && !empty($session_id)) {
      $query->condition('session_uuid', $session_id);
    }
    else {
      $query->condition('uid', $user_id);
    }
    
    $query->sort('created', 'DESC');
    $query->range(0,1);

    $results = $query->execute();

    if (!empty($results)) {
      if ($attempt = reset($storage->loadMultiple($results))) {        
        $last_attempt = $attempt->created->value;
      }      
    }

    return $last_attempt;

  }

  /**
   * Check if we allow new attempt
   *
   * @param EntityInterface $entity
   *   The entity we have attached attempt management.
   * @param integer $user_id
   *   The user for the attempt.
   * @param [type] $session_id
   *   The session_id if the user is anonymous.
   * @return bool
   *   TRUE | FALSE
   */
  public function allowNewAttempt(EntityInterface $entity, int $user_id, $session_id): bool {

    $allowed = TRUE;


    $attempts = $this->numberAttemptsPerUser($entity, $user_id, $session_id);

    $attempt_limit = $this->getAttemptFieldByProperty($entity, 'limit');
      
    if ($attempt_limit > 0) {
      if ($attempts == $attempt_limit) {
        $allowed = FALSE;
      }      
    }    

    return $allowed;

  }

  protected function createOrUpdateAttempt($field_data, $entity, $user_id, $session_id) {

    // If we do not have any attempt yet
    if ($this->numberAttemptsPerUser($entity, $user_id, $session_id) == 0) {

    }
    

    $storage = $this->entityTypeManager->getStorage('attempt_mgmt_attempt')->create($field_data);
    $storage->save();


  }

  public function getExistingTemporaryAttempt(EntityInterface $entity, $user_id) {

    // For anon. user we need to save the session id.
    if ($user_id == 0) {
      $session_id = $this->getSessionIDForUnauthenticatedUsers();
    }
    else {
      $session_id = NULL;
    } 

    $storage = $this->entityTypeManager->getStorage('attempt_mgmt_attempt');
    $query = $storage->getQuery();
    $query->condition('entity_type', $entity->getEntityTypeId())
          ->condition('entity_id', $entity->id())
          ->accessCheck(FALSE);
    
    // Search for session_id on anonymous users.
    if ($user_id == 0 && !empty($session_id)) {
      $query->condition('session_uuid', $session_id);
    }
    else {
      $query->condition('uid', $user_id);
    }

    $query->condition('temporary', TRUE);

    $results = $query->execute();

    if (!empty($results)) {
      $attempts = $storage->loadMultiple($results);
      if ($attempt = reset($attempts)) {
        return $attempt->uuid();
      }
    }

    return NULL;

  }

  /**
   * Set the current attempt to close
   *
   * @param string $attempt_uuid
   *   The uuid for the attempt.
   * @return void
   */
  public function setCurrentAttemptToClosed($attempt_uuid) {
    $storage = $this->entityTypeManager->getStorage('attempt_mgmt_attempt');
    $attempts = $storage->loadByProperties(['uuid' => $attempt_uuid]);
    if ($attempt = reset($attempts)) {
      //if ($attempt->temporary == FALSE && $attempt->closed == FALSE) {
        $attempt->closed = TRUE;
        $attempt->save();
      //}      
    }
  }

  /**
   * Get current attempt
   *
   * @param EntityInterface $entity
   *   The entity the attempt is attached to.
   * @param [type] $user_id
   *   The user id for the attempt.
   * @param [type] $session_id
   *   The session id if anon. user.
   * @return 
   *   string uuid | NULL
   */
  public function getCurrentAttempt(EntityInterface $entity, $user_id) {

    // For anon. user we need to save the session id.
    if ($user_id == 0) {
      $session_id = $this->getSessionIDForUnauthenticatedUsers();
    }
    else {
      $session_id = NULL;
    }  
    
    $storage = $this->entityTypeManager->getStorage('attempt_mgmt_attempt');
    $query = $storage->getQuery();
    $query->condition('entity_type', $entity->getEntityTypeId())
          ->condition('entity_id', $entity->id())
          ->accessCheck(FALSE);

    // Search for session_id on anonymous users.
    if ($user_id == 0 && !empty($session_id)) {
      $query->condition('session_uuid', $session_id);
    }
    else {
      $query->condition('uid', $user_id);
    }

    // We only want real attempts
    $query->condition('temporary', FALSE);

    // We only want open attempts
    //$query->condition('closed', FALSE);
    
    $query->sort('created', 'DESC');
    $query->range(0,1);

    $results = $query->execute();
    if (!empty($results)) {
      $attempts = $storage->loadMultiple($results);
      if ($attempt = reset($attempts)) {
        return $attempt->uuid();
      }
    }

    return NULL;

  }

  public function getLastAttemptForUser(EntityInterface $entity, $user_id) {
    $storage = $this->entityTypeManager->getStorage('attempt_mgmt_attempt');
    $query = $storage->getQuery();
    $query->condition('entity_type', $entity->getEntityTypeId())
          ->condition('entity_id', $entity->id())
          ->accessCheck(FALSE);

    // For anon. user we need to save the session id.
    if ($user_id == 0) {
      $session_id = $this->getSessionIDForUnauthenticatedUsers();
    }
    else {
      $session_id = NULL;
    }    

    // Search for session_id on anonymous users.
    if ($user_id == 0 && !empty($session_id)) {
      $query->condition('session_uuid', $session_id);
    }
    else {
      $query->condition('uid', $user_id);
    }

    $query->sort('id', 'DESC');
    $query->range(0,1);

    $results = $query->execute();
    if (!empty($results)) {
      $attempts = $storage->loadMultiple($results);
      if ($attempt = reset($attempts)) {
        return $attempt->uuid();
      }
    }

    return NULL;

  }
  /**
   * Get an existing attempt.
   *
   * @param EntityInterface $entity
   *   The entity the attempt is attached to.
   * @param [type] $user_id
   *   The user id of the attempt.
   * @param [type] $session_id
   *   The session id if anon. user.
   * @return bool
   *   TRUE | FALSE
   */
  public function getExistingAttempt(EntityInterface $entity, $user_id):bool {

    $existing_attempts = FALSE;

    // For anon. user we need to save the session id.
    if ($user_id == 0) {
      $session_id = $this->getSessionIDForUnauthenticatedUsers();
    }
    else {
      $session_id = NULL;
    }

    $storage = $this->entityTypeManager->getStorage('attempt_mgmt_attempt');
    $query = $storage->getQuery();
    $query->condition('entity_type', $entity->getEntityTypeId())
          ->condition('entity_id', $entity->id())
          ->accessCheck(FALSE);

    // Search for session_id on anonymous users.
    if ($user_id == 0 && !empty($session_id)) {
      $query->condition('session_uuid', $session_id);
    }
    else {
      $query->condition('uid', $user_id);
    } 

    $results = $query->execute();

    if (!empty($results)) {
      $existing_attempts = TRUE;
    }

    return $existing_attempts;

  }

  public function setAttemptsToClosedForUser(AccountInterface $account) {

    $storage = $this->entityTypeManager->getStorage('attempt_mgmt_attempt');
    $query = $storage->getQuery();
    $query->condition('uid', $account->id())
          ->condition('closed', FALSE)
          ->condition('temporary', FALSE)  
          ->accessCheck(FALSE);    

    $results = $query->execute();
    if (!empty($results)) {

      $records = count($results);
      if ($records > 0) {
        \Drupal::logger('attempt_mgmt')->notice(t('set @records attempts to closed for @user', [
          '@user' => $account->getDisplayName(),
          '@records' => $records, 
        ]));
      }
      $attempts = $storage->loadMultiple($results);
      if (!empty($attempts)) {
        foreach ($attempts as $attempt) {
          $attempt->closed = TRUE;
          $attempt->save();
        }
      }
    }
  }

  public function getAttemptConfig() {

    $attempt_config = [];

    $config = \Drupal::config('attempt_mgmt.settings');
    if (isset($config)) {
      $attempt_config['question'] = $config->get('attempt_question');
      $attempt_config['start_new_attempt_label'] = $config->get('start_new_attempt_label');
      $attempt_config['proceed_attempt_label'] = $config->get('proceed_attempt_label');      
    }
    return $attempt_config;
  }

}
