<?php

declare(strict_types=1);

namespace Drupal\attempt_mgmt\Entity;

use Drupal\attempt_mgmt\AttemptInterface;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\user\EntityOwnerTrait;
use Drupal\Core\Datetime\DrupalDateTime;

/**
 * Defines the attempt entity class.
 *
 * @ContentEntityType(
 *   id = "attempt_mgmt_attempt",
 *   label = @Translation("Attempt"),
 *   label_collection = @Translation("Attempts"),
 *   label_singular = @Translation("attempt"),
 *   label_plural = @Translation("attempts"),
 *   label_count = @PluralTranslation(
 *     singular = "@count attempts",
 *     plural = "@count attempts",
 *   ),
 *   bundle_label = @Translation("Attempt type"),
 *   handlers = {
 *     "list_builder" = "Drupal\attempt_mgmt\AttemptListBuilder",
 *     "views_data" = "Drupal\views\EntityViewsData",
 *     "form" = {
 *       "add" = "Drupal\attempt_mgmt\Form\AttemptForm",
 *       "edit" = "Drupal\attempt_mgmt\Form\AttemptForm",
 *       "delete" = "Drupal\Core\Entity\ContentEntityDeleteForm",
 *       "delete-multiple-confirm" = "Drupal\Core\Entity\Form\DeleteMultipleForm",
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\attempt_mgmt\Routing\AttemptHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "attempt_mgmt_attempt",
 *   admin_permission = "administer attempt_mgmt_attempt types",
 *   entity_keys = {
 *     "id" = "id",
 *     "bundle" = "bundle",
 *     "uuid" = "uuid",
 *     "owner" = "uid",
 *   },
 *   links = {
 *     "collection" = "/admin/content/attempt",
 *     "add-form" = "/attempt/add/{attempt_mgmt_attempt_type}",
 *     "add-page" = "/attempt/add",
 *     "canonical" = "/attempt/{attempt_mgmt_attempt}",
 *     "edit-form" = "/attempt/{attempt_mgmt_attempt}",
 *     "delete-form" = "/attempt/{attempt_mgmt_attempt}/delete",
 *     "delete-multiple-form" = "/admin/content/attempt/delete-multiple",
 *   },
 *   bundle_entity_type = "attempt_mgmt_attempt_type",
 *   field_ui_base_route = "entity.attempt_mgmt_attempt_type.edit_form",
 * )
 */
final class Attempt extends ContentEntityBase implements AttemptInterface {

  use EntityChangedTrait;
  use EntityOwnerTrait;

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage): void {
    parent::preSave($storage);
    if (!$this->getOwnerId()) {
      // If no owner has been set explicitly, make the anonymous user the owner.
      $this->setOwnerId(0);
    }
    if ($this->getCreatedTime() < $this->getChangedTime()) {
      $this->setDuration();
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function preCreate(EntityStorageInterface $storage, array &$values) {
    parent::preCreate($storage, $values);
    $values += [
      'ip' => \Drupal::request()->getClientIp(),
    ];
  }  

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {

    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['status'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Status'))
      ->setDefaultValue(FALSE)
      ->setSetting('on_label', 'Enabled')
      ->setDisplayOptions('form', [
        'type' => 'boolean_checkbox',
        'settings' => [
          'display_label' => FALSE,
        ],
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'boolean',
        'label' => 'above',
        'weight' => 0,
        'settings' => [
          'format' => 'enabled-disabled',
        ],
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['uid'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Author'))
      ->setSetting('target_type', 'user')
      ->setRevisionable(FALSE)
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => 60,
          'placeholder' => '',
        ],
        'weight' => 15,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'author',
        'weight' => 15,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['entity_type'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Entity type'))
      ->setDescription(t('The machine name of the host entity type this attempt is attached to.'))
      ->setSetting('max_length', '32');

    $fields['entity_id'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Entity ID'))
      ->setDescription(t('The ID of the entity this attempt is attached to.'))
      ->setSetting('unsigned', TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Attempt started'))
      ->setDescription(t('The time that the attempt was created.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'datetime_timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Attempt updated'))
      ->setDescription(t('The time that the attempt was last updated.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'datetime_timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['duration'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Duration'));

    $fields['temporary'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Temporary'))
      ->setDefaultValue(TRUE);      

    $fields['closed'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Closed'))
      ->setDefaultValue(FALSE);         

    $fields['number_attempt'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Number of attempt'))
      ->setDescription(t('The number of the users attempt.'))
      ->setSetting('unsigned', TRUE);

    $fields['ip'] = BaseFieldDefinition::create('string')
      ->setLabel(t('IP'));
      
    $fields['session_uuid'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Session UUID'));       

    return $fields;

  }

  /**
   * {@inheritdoc}
   */
  public function getSessionUuid() {
    return $this->get('session_uuid')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setSessionUuid($session_uuid) {
    $this->get('session_uuid')->setValue($session_uuid);
    return $this;
  }   

  public function getCreatedTime() {
    return $this->get('created')->value;
  }  

  public function setCreatedTime($timestamp) {
    $this->set('created', $timestamp);
    return $this;
  }  

  /**
   * {@inheritdoc}
   */  
  public function setDuration() {

    $created = $this->getCreatedTime();
    $changed = $this->getChangedTime();


    $date_formatter = \Drupal::service('date.formatter');
    $created_date = $date_formatter->format($created, 'custom', 'Y-m-d H:i:s');    
    $changed_date = $date_formatter->format($changed, 'custom', 'Y-m-d H:i:s');  

    $datetime_created = new \DateTime($created_date);
    $datetime_changed = new \DateTime($changed_date);
    $interval = $datetime_changed->diff($datetime_created);

    if (is_object($interval)) {

    }

    //$duration = $interval->format('%d days, %H hours, %I minutes, %S seconds');
    $duration = $interval->format('%D:%H:%I:%S');

    $this->get('duration')->setValue($duration);
    return $this;

  }

}
