<?php

declare(strict_types=1);

namespace Drupal\attempt_mgmt\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\Core\Config\ConfigFactory;
use Drupal\attempt_mgmt\AttemptFactory;
use Drupal\Core\Entity\EntityInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\InvokeCommand;
use Drupal\ajax_command_page_reload\Ajax\PageReloadCommand;


/**
 * @todo Add a description for the form.
 */
final class AttemptConfirmForm extends FormBase {

  /**
   * The config factory.
   * 
   * @var \Drupal\Core\Config\ConfigFactory
   */
  protected $configFactory;

  /**
   * The attempt factory.
   * 
   * @var \Drupal\attempt_mgmt\AttemptFactory
   */
  protected $attemptFactory;

  /**
   * The module handler.
   * 
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The entity type manager.
   * 
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity type id.
   * 
   * @var string 
   */
  protected $entityTypeId;

  /**
   * The entity id.
   * 
   * @var int 
   */
  protected $entityId;

  /**
   * The account.
   * 
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $account;  

  /**
   * The attempt uuid.
   * 
   * @var string attempt_uuid
   */
  protected $attemptUuid;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new static();
    $instance->setAttemptFactory($container->get('attempt_mgmt.factory'));
    $instance->setConfigFactory($container->get('config.factory'));
    $instance->setModuleHandler($container->get('module_handler'));
    $instance->setEntityTypeManager($container->get('entity_type.manager'));
    return $instance;
  }

  /**
   * Set the route match service.
   *
   * @param \Drupal\attempt_mgmt\AttemptFactory $attempt_factory
   *   The attempt factory service.
   */
  public function setAttemptFactory($attempt_factory) {
    $this->attemptFactory = $attempt_factory;
  }  

  /**
   * Set the config factory service.
   *
   * @param \Drupal\Core\Config\ConfigFactory $config_factory
   *   The config factory service.
   */
  public function setConfigFactory($config_factory) {
    $this->configFactory = $config_factory;
  }    

  /**
   * Set the module handler service.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   */
  public function setModuleHandler($module_handler) {
    $this->moduleHandler = $module_handler;
  }   

  /**
   * Set the entity type manager.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function setEntityTypeManager($entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }     

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'attempt_mgmt_attempt_confirm';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, string $entity_type_id = NULL, int $entity_id = NULL, $account = NULL) {
    
    // Get a session object
    $session = \Drupal::request()->getSession();

    // Assign incoming parameters to properties
    // to access them in every method of the form object.
    $this->entityTypeId = $entity_type_id;
    $this->entityId = $entity_id;    
    $this->account = $account;     


    $form['#attributes']['id'] = 'attempt-confirm-form-container';
    $form['#attributes']['class'][] = 'attempt-confirm-form';    

    // Get the entity 
    $entity = $this->entityTypeManager->getStorage($entity_type_id)->load($entity_id);
    if ($entity instanceof EntityInterface) {
      // First of all check if we have force new attempt on.
      if ($this->attemptFactory->isForceNewAttempt($entity)) {
        // Make sure the form is hidden
        $form['#attributes']['class'][] = 'hidden';
        // Create a new attempt
        $attempt_uuid = $this->attemptFactory->createAttempt($entity, $account->id());
      }
      else {
        // First we check if we already have an attempt for that user
        $attempt_exists = $this->attemptFactory->getExistingAttempt($entity, $account->id());

        // If not we want to hide our attempt form and create an attempt to start
        if (!$attempt_exists) {
          $session->set('hide_form', TRUE); 
          $attempt_uuid = $this->attemptFactory->createAttempt($entity, $account->id());           
        }
        else {    
          // In case we do have already attempts check for a temporary attempt
          if ($uuid = $this->attemptFactory->getExistingTemporaryAttempt($entity, $account->id())) {
            $attempt_uuid = $uuid;
          }
          elseif ($uuid = $this->attemptFactory->getCurrentAttempt($entity, $account->id())) {
            $attempt_uuid = $uuid;
          }
        }
      }
    }

    if (isset($attempt_uuid)) {
      $attempt_uuid_default = $attempt_uuid;      
    }
    else {
      $attempt_uuid_default = 'NONE'; 
    }

    // Get the session data
    $data = $session->get('hide_form');

    if(isset($data) && $data == TRUE) {
      $form['#attributes']['class'][] = 'hidden';
      $session->set('hide_form', FALSE);
    }    
    
    $this->attemptUuid = $attempt_uuid_default;

    $form['attempt_uuid'] = [
      '#attributes' => ['id' => 'attempt_mgmt_uuid'],
      '#type' => 'hidden',
      '#value' => $attempt_uuid_default,
    ];

    $form['question']['#markup'] = $this->getQuestion();

    $form['actions']['#type'] = 'actions';

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->getConfirmText(),
      '#button_type' => 'primary',
      '#attributes' => ['class' => ['confirm-form-button']], 
    ];

    $form['actions']['cancel'] = [
      '#type' => 'submit',
      '#value' => $this->getCancelText(),
      '#button_type' => 'default',
      '#attributes' => ['class' => ['confirm-form-button']], 
    ]; 

    $form['description']['#prefix'] = '<p>';
    $form['description']['#suffix'] = '</p>';

    $form['#cache'] = ['max-age' => 0];

    $form['#attached']['library'][] = 'attempt_mgmt/confirm-form-design';
    
    return $form; 

  }  

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    $attempt_question = $this->configFactory->get('attempt_mgmt.settings')->get('attempt_question');
    if (isset($attempt_question)) {
      return $attempt_question;
    }
    return $this->t('Do you want to start a new attempt?');    
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl(): Url {
    $attempt_uuid = $this->attemptUuid;
    // Update the current attempt to closed!
    if (isset($attempt_uuid) && !$attempt_uuid === 'NONE') {
      $this->attemptFactory->setCurrentAttemptToClosed($attempt_uuid);
    }
    return new Url('<current>');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {      
    $start_new_attempt = $this->configFactory->get('attempt_mgmt.settings')->get('start_new_attempt_label');
    if (isset($start_new_attempt)) {
      return $start_new_attempt;
    }
    return $this->t('Confirm');
  }
  
  /**
   * {@inheritdoc}
   */
  public function getCancelText() {
    $proceed_attempt = $this->configFactory->get('attempt_mgmt.settings')->get('proceed_attempt_label');
    if (isset($proceed_attempt)) {
      return $proceed_attempt; 
    }  
    return $this->t('Cancel');
  }  


  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Get a session object
    $session = \Drupal::request()->getSession();

    // Get uuid
    $attempt_uuid = $form_state->getValue('attempt_uuid');

    $triggered_element = $form_state->getTriggeringElement();
    $action = $triggered_element['#id'];

    if ($action === 'edit-submit') {

      $session->set('hide_form', TRUE);
      /** @var \Drupal\Core\TempStore\PrivateTempStore $store */
      $store = \Drupal::service('tempstore.private')->get('attempt_mgmt');
      $store->set('hide_form', TRUE);

      // To give other modules the chance to work on a new attempt.
      if (isset($this->entityTypeId) && isset($this->entityId)) {
        $entity = $this->entityTypeManager->getStorage($this->entityTypeId)
          ->load($this->entityId);           
        if ($entity instanceof EntityInterface) {
          // Account
          $account = $this->account;
          $this->moduleHandler->invokeAll('attempt_mgmt_new_attempt', [
            $entity,
            $account
          ]);
        }
      }
      // Create new attempt
      $this->attemptFactory->createAttempt($entity, $account->id());
    }

    if ($action === 'edit-cancel') { 
      $session->set('hide_form', TRUE);
    }

  }

  /**
   * Callback for submit_driven example.
   *
   * Select the 'box' element, change the markup in it, and return it as a
   * renderable array.
   *
   * @return array
   *   Renderable array (the box element)
   */
  /*
   public function cancelCallback(array &$form, FormStateInterface $form_state) {
    // In most cases, it is recommended that you put this logic in form
    // generation rather than the callback. Submit driven forms are an
    // exception, because you may not want to return the form at all.
    $form_state->setRebuild(FALSE);
    $element = $form;
    $element['#attributes']['class'][] = 'hidden'; 
    return $element;
  }
    */



}

