<?php

declare(strict_types=1);

namespace Drupal\attempt_mgmt\Plugin\Field\FieldType;

use Drupal\Component\Utility\Random;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemBase;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\TypedData\DataDefinition;
use Drupal\attempt_mgmt\Entity\AttemptType;
use Drupal\attempt_mgmt\Utility\FieldOverride;
use Drupal\attempt_mgmt\Utility\FieldOverrides;
use Drupal\attempt_mgmt\FieldHelper;
use Drupal\Core\Messenger\MessengerTrait;
use Drupal\Core\Url;
use Drupal\Core\Link;

/**
 * Defines the 'attempt_mgmt_attempt_management' field type.
 *
 * @FieldType(
 *   id = "attempt_mgmt_attempt_settings",
 *   label = @Translation("Attempt Settings"),
 *   description = @Translation("Attempt Settings to control the attempt."),
 *   default_widget = "attempt_mgmt_settings_default",
 *   default_formatter = "attempt_mgmt_settings_default",
 *   cardinality = 1
 * )
 */
final class AttemptManagementItem extends FieldItemBase {

  use MessengerTrait;

  /**
   * {@inheritdoc}
   */
  //public static function defaultStorageSettings(): array {
    //$settings = ['foo' => ''];
    //return $settings + parent::defaultStorageSettings();
  //}

  /**
   * {@inheritdoc}
   */
  // public function storageSettingsForm(array &$form, FormStateInterface $form_state, $has_data): array {
  //   $element['foo'] = [
  //     '#type' => 'textfield',
  //     '#title' => $this->t('Foo'),
  //     '#default_value' => $this->getSetting('foo'),
  //     '#disabled' => $has_data,
  //   ];
  //   return $element;
  // }

  /**
   * {@inheritdoc}
   */
  public static function defaultFieldSettings() {
    return [
      'field_overrides' => [],
      // Replaced by field_overrides.
      'fields' => [],
      'attempt_type' => '',
    ] + parent::defaultFieldSettings();
  }  

  /**
   * {@inheritdoc}
   */
  // public function isEmpty(): bool {
  //   return match ($this->get('value')->getValue()) {
  //     NULL, '' => TRUE,
  //     default => FALSE,
  //   };
  // }

  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition): array {

    // @DCG
    // See /core/lib/Drupal/Core/TypedData/Plugin/DataType directory for
    // available data types.

    $properties['attempt_type'] = DataDefinition::create('string')
      ->setLabel(t('Attempt type'))
      ->setRequired(TRUE);

    $properties['enabled'] = DataDefinition::create('boolean')
      ->setLabel(t('Enable Attempt management'));

    $properties['limit'] = DataDefinition::create('integer')
      ->setLabel(t('Limit attempt'))
      ->setRequired(TRUE);

    $properties['force_new_attempt'] = DataDefinition::create('boolean')
      ->setLabel(t('Force new attempt'))
      ->setDescription(t('Force new attempt when hitting scorm.'))
      ->setRequired(1);

    $properties['grading_method'] = DataDefinition::create('string')
      ->setLabel(t('Attempt grading method'))
      ->setRequired(TRUE);

    $properties['display_status'] = DataDefinition::create('boolean')
      ->setLabel(t('Attempt display status'));

    $properties['attempt_confirm_delay'] = DataDefinition::create('integer')
      ->setLabel(t('Attempt confirm delay'))
      ->setRequired(TRUE);

    $properties['lock'] = DataDefinition::create('boolean')
      ->setLabel(t('Lock after final attempt'))
      ->setRequired(FALSE);

    return $properties;

  }

  // /**
  //  * {@inheritdoc}
  //  */
  // public function getConstraints(): array {
  //   $constraints = parent::getConstraints();

  //   $constraint_manager = $this->getTypedDataManager()->getValidationConstraintManager();

  //   // @DCG Suppose our value must not be longer than 10 characters.
  //   $options['value']['Length']['max'] = 10;

  //   // @DCG
  //   // See /core/lib/Drupal/Core/Validation/Plugin/Validation/Constraint
  //   // directory for available constraints.
  //   $constraints[] = $constraint_manager->create('ComplexData', $options);
  //   return $constraints;
  // }

  /**
   * {@inheritdoc}
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition): array {

    $columns = [
      'attempt_type' => [
        'type' => 'varchar',
        'not null' => TRUE,
        'description' => 'The attempt type.',
        'length' => 32,
      ],
      'enabled' => [
        'type' => 'int',
        'size' => 'tiny',
        'unsigned' => TRUE,
        'not null' => TRUE,
        'description' => 'Whether the Attempt management is enabeld.',
      ],
      'force_new_attempt' => [
        'type' => 'int',
        'size' => 'tiny',
        'unsigned' => TRUE,
        'not null' => FALSE,
        'description' => 'Whether the attempt should start at hitting scorm.',
      ],      
      'limit' => [
        'type' => 'int',
        'size' => 'tiny',
        'unsigned' => TRUE,
        'not null' => TRUE,
        'description' => 'The limit of the attempt.',
      ],
      'grading_method' => [
        'type' => 'varchar',
        'not null' => TRUE,
        'description' => 'Grading method for attempt.',
        'length' => 255,
      ],
      'display_status' => [
        'type' => 'int',
        'size' => 'tiny',
        'unsigned' => TRUE,
        'not null' => TRUE,
        'description' => 'Display status of attempt.',
      ],
      'attempt_confirm_delay' => [
        'type' => 'int',
        'size' => 'small',
        'unsigned' => TRUE,
        'not null' => TRUE,
        'description' => 'Attempt confirm delay.',
      ],
      'lock' => [
        'type' => 'int',
        'size' => 'tiny',
        'unsigned' => TRUE,
        'not null' => TRUE,
        'description' => 'Lock after final attempt.',
      ],
    ];

    $schema = [
      'columns' => $columns,
      // @todo Add indexes here if necessary.
    ];

    return $schema;
  }


  /**
   * {@inheritdoc}
   */
  public function getAttemptType(): string {
    return $this->getSetting('attempt_type');
  }

  /**
   * {@inheritdoc}
   */
  public function getEnabled(): bool {
    return $this->enabled ?? FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function getForceNewAttempt(): bool {
    return $this->force_new_attempt ?? FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function getLimit(): int {
    return $this->limit ?? 0;
  }

  /**
   * {@inheritdoc}
   */
  public function getGradingMethod(): string {
    return $this->grading_method ?? 'best_attempt';
  }

  /**
   * {@inheritdoc}
   */
  public function getDisplayStatus(): bool {
    return $this->display_status ?? TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getAttemptConfirmDelay(): int {
    return $this->attempt_confirm_delay ?? 0;
  }  

  /**
   * {@inheritdoc}
   */
  public function getLock(): bool {
    return $this->lock ?? FALSE;
  }

  /**
   * Gets the field overrides for the current field.
   *
   * @return array
   *   FieldOverride constants keyed by AddressField constants.
   */
  public function getFieldOverrides() {
    $field_overrides = [];
    if ($fields = $this->getSetting('fields')) {
      $unused_fields = array_diff($this->getAllFields(), $fields);
      foreach ($unused_fields as $field) {
        $field_overrides[$field] = FieldOverride::HIDDEN;
      }
    }
    elseif ($overrides = $this->getSetting('field_overrides')) {
      foreach ($overrides as $field => $data) {
        $field_overrides[$field] = $data['override'];
      }
    }

    return $field_overrides;
  }

  public function getAllFields() {

    $fields = [      
      //'enabled' => ['label' => $this->t('Enable attempt management'), 'value' => null],
      'limit' => ['label' => $this->t('Limit attempts'), 'value' => null],
      'force_new_attempt' => ['label' => $this->t('Force new attempt'), 'value' => null],
      'grading_method' => ['label' => $this->t('Grading method'), 'value' => null],
      'display_status' => ['label' => $this->t('Display status'), 'value' => null],
      'attempt_confirm_delay' => ['label' => $this->t('Attempt confirm delay'), 'value' => null],
      'lock' => ['label' => $this->t('Lock after final attempt'), 'value' => null],      
    ];

    return $fields;

  }

  protected function getGenericFieldLabels() {

    return [
      //'enabled' => t('Enable attempt management', [], ['context' => 'Attempt Settings label']),
      'limit' => t('Limit attempts', [], ['context' => 'Attempt Settings label']),
      'force_new_attempt' => t('Force new attempt', [], ['context' => 'Attempt Settings label']),
      'grading_method' => t('Grading method', [], ['context' => 'Attempt Settings label']),
      'display_status' => t('Display status', [], ['context' => 'Attempt Settings label']),
      'attempt_confirm_delay' => t('Attempt confirm delay', [], ['context' => 'Attempt Settings label']),
      'lock' => t('Lock after final attempt', [], ['context' => 'Attempt Settings label']),      
    ];


  }  

  /**
   * {@inheritdoc}
   */
  public function getProperties($include_computed = FALSE) {
    $properties = parent::getProperties($include_computed);
    $parsed_overrides = new FieldOverrides($this->getFieldOverrides());
    $hidden_properties = array_map(static function ($name) {
      return FieldHelper::getPropertyName($name);
    }, $parsed_overrides->getHiddenFields());
    foreach ($hidden_properties as $hidden_property) {
      unset($properties[$hidden_property]);
    }
    return $properties;
  }  

  /**
   * Form element validation handler: Removes empty field overrides.
   *
   * @param array $element
   *   The field overrides form element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state of the entire form.
   */
  public static function fieldOverridesValidate(array $element, FormStateInterface $form_state) {
    $overrides = $form_state->getValue($element['#parents']);
    $overrides = array_filter($overrides, function ($data) {
      return !empty($data['override']);
    });
    $form_state->setValue($element['#parents'], $overrides);
  }

  /**
   * {@inheritdoc}
   */
  public function fieldSettingsForm(array $form, FormStateInterface $form_state) {

    $attempt_type_options = [];

    $attempt_types = \Drupal::entityTypeManager()->getStorage('attempt_mgmt_attempt_type')->loadMultiple();
    foreach ($attempt_types as $machine_name => $type) {
      $attempt_type_options[$machine_name] = $type->label();
    }

    $destination = \Drupal::destination()->getAsArray();
    $url = Url::fromRoute('entity.attempt_mgmt_attempt_type.add_form', [], ['query' => $destination]);
    $link = Link::fromTextAndUrl(t('Add attempt type'), $url)->toString();
    
    if (empty($attempt_type_options)) {
      $this->messenger()->addWarning($this->t('You have to create at least one attempt type! @link', ['@link' => $link]));      
    }


    $element['attempt_type'] = [
      '#type' => 'radios',
      '#title' => $this->t('Attempt type'),
      '#options' => $attempt_type_options,  
      '#required' => TRUE,  
      '#default_value' => $this->getSetting('attempt_type'),
    ];

    $element['field_overrides_title'] = [
      '#type' => 'item',
      '#title' => $this->t('Field overrides'),
      '#description' => $this->t('Use field overrides to override the Big Blue Button settings, forcing specific properties to always be hidden, optional, or required.'),
    ];

    $element['field_overrides'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Property'),
        $this->t('Override'),
      ],
      '#element_validate' => [[get_class($this), 'fieldOverridesValidate']],
    ];

    $field_overrides = $this->getFieldOverrides();
    
    foreach ($this->getGenericFieldLabels() as $field_name => $label) {
      $override = $field_overrides[$field_name] ?? '';

      $element['field_overrides'][$field_name] = [
        'field_label' => [
          '#type' => 'markup',
          '#markup' => $label,
        ],
        'override' => [
          '#type' => 'select',
          '#options' => [
            FieldOverride::HIDDEN => $this->t('Hidden'),
            //FieldOverride::OPTIONAL => $this->t('Optional'),
            //FieldOverride::REQUIRED => $this->t('Required'),
          ],
          '#default_value' => $override,
          '#empty_option' => $this->t('- No override -'),
        ],
      ];
    }

    return $element;


  }









}
