<?php declare(strict_types = 1);

namespace Drupal\attempt_mgmt\Plugin\Field\FieldWidget;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Render\Element;
use Drupal\Core\Link;
use Drupal\attempt_mgmt\Utility\AttemptMgmtField;

/**
 * Defines the 'attempt_mgmt_settings_default' field widget.
 *
 * @FieldWidget(
 *   id = "attempt_mgmt_settings_default",
 *   label = @Translation("Attempt management default"),
 *   field_types = {
 *     "attempt_mgmt_attempt_settings"
 *   },
 * )
 */
final class AttemptMgmtSettingsDefaultWidget extends WidgetBase implements ContainerFactoryPluginInterface {

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;  

  /**
   * Constructs a AddressDefaultWidget object.
   *
   * @param string $plugin_id
   *   The plugin_id for the widget.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the widget is associated.
   * @param array $settings
   *   The widget settings.
   * @param array $third_party_settings
   *   Any third party settings.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, array $third_party_settings, ConfigFactoryInterface $config_factory) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings);
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new self(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {

    $attempt_type_options = [];

    $entity = $items[$delta]->getEntity();        
    $item = $items[$delta];
    $value = $item->toArray();

    // Default values
    $enabled = $items[$delta]->enabled ?? TRUE;
    $attempt_type = $items[$delta]->getAttemptType();
    $force_new_attempt = $items[$delta]->force_new_attempt ?? TRUE;
    $limit = intval($items[$delta]->limit) ?? 0;
    $grading_method = $items[$delta]->grading_method ?? 'best_attempt';
    $display_status = $items[$delta]->display_status ?? TRUE;
    $attempt_confirm_delay = intval($items[$delta]->attempt_confirm_delay) ?? 0;
    $lock = $items[$delta]->lock ?? TRUE;

    //dump($attempt_type);

    $field_name = $items[$delta]->getFieldDefinition()->getName();
    $input_name = $field_name . "[0][enabled]";

    $element += [
      '#type' => 'details',
      '#open' => TRUE,
      '#description' => t('Attempt Management'),
    ];    

    $element['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable attempt management'),
      '#default_value' => $enabled,
      '#description' => $this->t('Tick to enable attempt management.'),
    ];     
    
    $attempt_types = \Drupal::entityTypeManager()->getStorage('attempt_mgmt_attempt_type')->loadMultiple();
    foreach ($attempt_types as $machine_name => $type) {
      $attempt_type_options[$machine_name] = $type->label();
    }    

    $element['attempt_type'] = [
      '#type' => 'textfield',
      '#default_value' => $attempt_type,
      '#disabled' => TRUE,
      '#title' => $this->t('Attempt type'),
    ];

    $element['force_new_attempt'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Force new attempt'),
      '#default_value' => $force_new_attempt,
      '#description' => $this->t('Forces new attempt on visit.'),
    ];   

    $element['limit'] = [
      '#type' => 'number',
      '#min' => 0,
      '#max' => 100,
      '#title' => $this->t('Limit attempts'),
      '#default_value' => $limit,
      '#description' => $this->t('Limit the attempts of an user.'),
      '#states' => [
        'visible' => [
          ":input[name='$input_name']" => ['checked' => TRUE],
        ]
      ]
    ];  

    $grading_method_options = [
      'best_attempt' => $this->t('Best attempt'),
      'last_attempt' => $this->t('Last attempt'),
    ];

    $element['grading_method'] = [
      '#type' => 'select',
      '#title' => $this->t('Grading method'),
      '#options' => $grading_method_options,
      '#default_value' => $grading_method,
      '#description' => $this->t('Grading method when multiple attempts allowed.'),
      '#states' => [
        'visible' => [
          ":input[name='$input_name']" => ['checked' => TRUE],
        ]
      ]
    ];

    $element['display_status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Display status'),
      '#default_value' => $display_status,
      '#description' => $this->t('Display status for attempt.'),
      '#states' => [
        'visible' => [
          ":input[name='$input_name']" => ['checked' => TRUE],
        ]
      ]
    ];  

    $element['attempt_confirm_delay'] = [
      '#type' => 'number',
      '#min' => 0,
      '#max' => 60000,
      '#title' => $this->t('Attempt confirm delay'),
      '#default_value' => $attempt_confirm_delay,
      '#description' => $this->t('Sometimes a delay before presenting a new attempt confirm form is needed.'),
      '#states' => [
        'visible' => [
          ":input[name='$input_name']" => ['checked' => TRUE],
        ]
      ]
    ];   

    $element['lock'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Lock'),
      '#default_value' => $lock,
      '#description' => $this->t('Lock after final attempt.'),
      '#states' => [
        'visible' => [
          ":input[name='$input_name']" => ['checked' => TRUE],
        ]
      ]
    ]; 

    $field_overrides = $items[$delta]->getFieldOverrides();

    foreach ($field_overrides as $key => $type) {
      if ($type === 'hidden') {
        $element[$key]['#access'] = FALSE;
      }
      elseif ($type === 'required') {
        $element[$key]['#required'] = TRUE;
      }
    }
  
    return $element;

  }

    /**
   * Validate the welcome message text field.
   */
  public function validateWelcomeMessage($element, FormStateInterface $form_state) {
    //$value = $element['#value'];
   
    
    // if (strlen($value) === 0) {
    //   $form_state->setValueForElement($element, '');
    //   return;
    // }

  }

  /**
   * Validate the logout url field.
   */
  public function validateLogoutUrl($element, FormStateInterface $form_state) {
    // $value = $element['#value'];
   
    
    // if (strlen($value) === 0) {
    //   $form_state->setValueForElement($element, '');
    //   return;
    // }

  }

  /**
   * Form API callback: Makes all address field properties optional.
   */
  public static function makeFieldsOptional(array $element, FormStateInterface $form_state) {
    foreach (Element::getVisibleChildren($element) as $key) {
      if (!empty($element[$key]['#required'])) {
        $element[$key]['#required'] = FALSE;
      }
    }
    return $element;
  } 

  /**
   * Reconstruct the values because of our multiple 
   * field properties.
   */  
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state) {

    // Get all field properties
    $field_properties = [];


    $new_values = [];
    foreach ($values as $delta => $value) {
      if (is_array($value)) {
        foreach ($value as $key => $property_value) {
          if (AttemptMgmtField::exists($key)) {
            $field_properties[$key] = $property_value;
          }
        }
      }        
      $new_values[$delta] = $field_properties;     
    }

    return $new_values;

  } 

}
