<?php

namespace Drupal\auctions_core\Plugin\QueueWorker;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\auctions_core\AuctionToolsTrait;

/**
 * Processes tasks for example module.
 *
 * @QueueWorker(
 *   id = "auction_item_workflow",
 *   title = @Translation("Auction Item Workflow update as by start/end dates"),
 *   cron = {"time" = 60}
 * )
 */
class AuctionItemWorkflow extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  use AuctionToolsTrait;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The logger service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * Constructs a new class instance.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   Entity type manager service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager, LoggerChannelFactoryInterface $logger_factory) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
    $this->loggerFactory = $logger_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('logger.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($item) {
    $entity = $this->entityTypeManager->getStorage('auction_item')->load($item->id);
    // Auction Item may have been deleted vie inline form 'remove'.
    if ($entity) {
      $entity->setWorkflow($item->workflow);
      $entity->save();
      $workflow = $this->itemWorkflows();
      $this->loggerFactory->get('Auctions Core')->notice('Auction Item ' . $item->id . ' workflow updated to ' . $workflow[$item->workflow]);
    }
  }

}
