<?php

namespace Drupal\audio_clips\Entity;

use Drupal\audio_clips\AudioClipTypeInterface;
use Drupal\Core\Config\Entity\ConfigEntityBundleBase;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\File\FileSystemInterface;

/**
 * Defines the Audio Clip type configuration entity.
 *
 * @ConfigEntityType(
 *   id = "audio_clip_type",
 *   label = @Translation("Audio Clip type"),
 *   handlers = {
 *     "form" = {
 *       "default" = "Drupal\audio_clips\Form\AudioClipTypeForm",
 *       "add" = "Drupal\audio_clips\Form\AudioClipTypeForm",
 *       "edit" = "Drupal\audio_clips\Form\AudioClipTypeForm",
 *       "delete" = "Drupal\audio_clips\Form\AudioClipTypeDeleteForm"
 *     },
 *     "list_builder" = "Drupal\audio_clips\Controller\AudioClipTypeListBuilder",
 *   },
 *   admin_permission = "administer clip types",
 *   config_prefix = "type",
 *   bundle_of = "audio_clip",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *   },
 *   links = {
 *     "delete-form" = "/admin/config/media/audio_clips/manage/{audio_clip_type}/delete",
 *     "edit-form" = "/admin/config/media/audio_clips/manage/{audio_clip_type}",
 *   },
 *    config_export = {
 *     "id",
 *     "label",
 *     "description",
 *     "min_duration",
 *     "max_duration",
 *   }
 * )
 */
class AudioClipType extends ConfigEntityBundleBase implements AudioClipTypeInterface {

  /**
   * The machine name of this audio clip type.
   *
   * @var string
   */
  public $id;

  /**
   * The human-readable name of the audio clip type.
   *
   * @var string
   */
  public $label;

  /**
   * A brief description of this audio clip type.
   *
   * @var string
   */
  public $description;

  /**
   * The audio clip minimum duration in seconds.
   *
   * @var int
   */
  public $min_duration;

  /**
   * The audio clip maximum duration in seconds.
   *
   * @var int
   */
  public $max_duration;

  /**
   * {@inheritdoc}
   */
  public function id() {
    return $this->id;
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return $this->description;
  }

  /**
   * {@inheritdoc}
   */
  public function getMinDuration() {
    return $this->min_duration;
  }

  /**
   * {@inheritdoc}
   */
  public function getMaxDuration() {
    return $this->max_duration;
  }

  /**
   * {@inheritdoc}
   */
  public static function getAllClipType() {
    return \Drupal::service('entity_type.manager')->getStorage('audio_clip_type')->loadMultiple();
  }

  /**
   * {@inheritdoc}
   */
  public static function getClipTypeNames() {
    return array_map(
      function ($bundle_info) {
        return $bundle_info['label'];
      },
      \Drupal::service('entity_type.bundle.info')->getBundleInfo('audio_clip')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function postSave(EntityStorageInterface $storage, $update = TRUE) {
    parent::postSave($storage, $update);

    $directory_target = 'public://audio_clip/clip_' . $this->id();
    \Drupal::service('file_system')->prepareDirectory($directory_target, FileSystemInterface::CREATE_DIRECTORY);
  }

  /**
   * {@inheritdoc}
   */
  public static function preDelete(EntityStorageInterface $storage, array $entities) {
    parent::postDelete($storage, $entities);
    foreach ($entities as $entity) {
      $directory_target = 'public://audio_clip/clip_' . $entity->id();
      \Drupal::service('file_system')->deleteRecursive($directory_target);
    }
  }

}
