<?php

namespace Drupal\audio_clips\Services;

use Drupal\audio_clips\Entity\AudioClip;
use Drupal\audio_clips\FFMpeg\AudioFile;
use Drupal\Core\Entity\EntityStorageException;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileExists;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\file\Entity\File;
use Drupal\file\FileUsage\FileUsageInterface;
use Drupal\user\Entity\User;

/**
 * Service to manage audio-clip creation.
 */
class AudioClipService {

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The clip storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected EntityStorageInterface $audioClipStorage;

  /**
   * The current User.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected AccountInterface $currentUser;

  /**
   * The file system.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected FileSystemInterface $fileSystem;

  /**
   * The file usage service.
   *
   * @var \Drupal\file\FileUsage\FileUsageInterface
   */
  protected FileUsageInterface $fileUsage;

  /**
   * AudioClipService constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager interface.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The account interface.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system interface.
   * @param \Drupal\file\FileUsage\FileUsageInterface $file_usage
   *   The file usage interface.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    AccountInterface $current_user,
    FileSystemInterface $file_system,
    FileUsageInterface $file_usage,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->audioClipStorage = $this->entityTypeManager->getStorage('audio_clip');
    $this->currentUser = $current_user;
    $this->fileSystem = $file_system;
    $this->fileUsage = $file_usage;
  }

  /**
   * Create audio-clip entity.
   *
   * @param int $fid
   *   The file ID.
   * @param int $mid
   *   The media ID.
   * @param string $clip_name
   *   The clip name.
   * @param int $start_time
   *   The start time in seconds.
   * @param int $end_time
   *   The end time in seconds.
   *
   * @return void
   *   Return void.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function createAudioClip(int $fid, int $mid, string $clip_name, int $start_time, int $end_time): void {
    $file_entity = $this->createAudioFileEntity($fid, $clip_name, $start_time, $end_time);

    $values = [
      'type' => $clip_name,
      'original_id' => $fid,
      'target_id' => $file_entity->id(),
      'start_time' => $start_time,
      'end_time' => $end_time,
    ];

    $audio_clip = $this->audioClipStorage->create($values);
    $audio_clip->save();
  }

  /**
   * Update audio-clip entity.
   *
   * @param int $fid
   *   The file ID.
   * @param int $mid
   *   The media ID.
   * @param string $clip_name
   *   The clip name.
   * @param int $start_time
   *   The start time in seconds.
   * @param int $end_time
   *   The end time in seconds.
   * @param \Drupal\audio_clips\Entity\AudioClip $audio_clip_storage
   *   The audio clip storage.
   *
   * @return void
   *   Return void.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function udpateAudioClip(int $fid, int $mid, string $clip_name, int $start_time, int $end_time, AudioClip $audio_clip_storage): void {
    $file_entity = $this->createAudioFileEntity($fid, $clip_name, $start_time, $end_time);

    $this->fileUsage->add($file_entity, 'file', 'media', $mid);

    $audio_clip_storage->set('start_time', $start_time);
    $audio_clip_storage->set('end_time', $end_time);
    $audio_clip_storage->set('entity_id', $file_entity->id());
    $audio_clip_storage->set('uri', $file_entity->getFileUri());
    $audio_clip_storage->setNewRevision();

    // @todo See to use method from revision as Node done
    $audio_clip_storage->setRevisionUserId(User::load($this->currentUser->id()));
    $audio_clip_storage->setRevisionLogMessage('Change start_time to ' . $start_time);

    $audio_clip_storage->save();
  }

  /**
   * Create new audio file entity.
   *
   * @param int $fid
   *   The file ID.
   * @param string $clip_name
   *   The clip name.
   * @param int $start_time
   *   The start time of the audio clip.
   * @param int $end_time
   *   The end time of the audio clip.
   *
   * @return \Drupal\file\Entity\File
   *   Return entity file created.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  private function createAudioFileEntity(int $fid, string $clip_name, int $start_time, int $end_time): File {
    $file_upload = File::load($fid);

    if (!$file_upload) {
      throw new EntityStorageException("The file with ID $fid could not be found.");
    }

    $file_upload_path = $this->fileSystem->realpath($file_upload->getFileUri());
    $audio_file = new AudioFile($file_upload_path);

    $directory_target = 'public://audio_clip/clip_' . $clip_name . '/public/' . date('Y-m');
    $this->fileSystem->prepareDirectory($directory_target, FileSystemInterface::CREATE_DIRECTORY);
    $file_path = $this->fileSystem->getDestinationFilename($directory_target . '/' . $file_upload->getFilename(), FileExists::Replace);
    $real_file_path = $this->fileSystem->realpath($file_path);

    $return_code = $audio_file->processClipAudio($start_time, $end_time, $real_file_path);
    if ($return_code !== 0) {
      throw new \RuntimeException("Error ffmpeg : " . $return_code);
    }

    $file_entity = File::create([
      'uri' => $file_path,
      'status' => 1,
    ]);
    $file_entity->save();

    return $file_entity;
  }

}
