<?php

namespace Drupal\audio_clips\Form;

use Drupal\Core\Entity\EntityConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form for audio clip type deletion.
 *
 * @internal
 */
class AudioClipTypeDeleteForm extends EntityConfirmFormBase {

  /**
   * String translation manager.
   *
   * @var \Drupal\Core\StringTranslation\TranslationInterface
   */
  protected $translation;

  /**
   * Constructs a new AudioClipTypeDeleteForm object.
   *
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   The string translation manager.
   */
  public function __construct(TranslationInterface $string_translation) {
    $this->translation = $string_translation;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('string_translation')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return t('Are you sure you want to delete the audio clip type %type?', ['%type' => $this->entity->label()]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('audio_clip.overview_types');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return t('Delete');
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $count = $this->entityTypeManager->getStorage('audio_clip')->getQuery()
      ->condition('type', $this->entity->id())
      ->accessCheck(TRUE)
      ->count()
      ->execute();
    if ($count) {
      $form['#title'] = $this->getQuestion();
      $form['description'] = [
        '#prefix' => '<p>',
        '#markup' => $this->translation->formatPlural($count, '%type is used by 1 piece of content on your site. You can not remove this content type until you have removed all of the %type content.', '%type is used by @count pieces of content on your site. You may not remove %type until you have removed all of the %type content.', ['%type' => $this->entity->label()]),
        '#suffix' => '</p>',
      ];
      return $form;
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->entity->delete();
    $t_args = ['%name' => $this->entity->label()];
    $this->messenger()->addMessage($this->t('The audio clip type %name has been deleted.', $t_args));
    $this->logger('audio_clip')->notice('Deleted audio clip type %name.', $t_args);

    $form_state->setRedirectUrl($this->getCancelUrl());
  }

}
