<?php

namespace Drupal\audio_clips\Form;

use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form handler for audio clip type forms.
 *
 * @internal
 */
class AudioClipTypeForm extends EntityForm {

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * Constructs the AudioTypeForm object.
   *
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager.
   */
  public function __construct(EntityFieldManagerInterface $entity_field_manager) {
    $this->entityFieldManager = $entity_field_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_field.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);

    $type = $this->entity;

    $form['label'] = [
      '#title' => $this->t('Label'),
      '#type' => 'textfield',
      '#default_value' => $type->label(),
      '#description' => $this->t('The human-readable name of this audio clip type. This name must be unique.'),
      '#required' => TRUE,
      '#size' => 30,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $type->id(),
      '#machine_name' => [
        'exists' => ['\Drupal\audio_clips\Entity\AudioClipType', 'load'],
        'source' => ['label'],
      ],
      // A audio clip type's machine name cannot be changed.
      '#disabled' => !$type->isNew(),
      '#description' => $this->t('A unique machine-readable name for this audio clip type. It must only contain lowercase letters, numbers, and underscores.'),
    ];

    $form['description'] = [
      '#title' => $this->t('Description'),
      '#type' => 'textarea',
      '#default_value' => $type->getDescription(),
      '#description' => $this->t('Displays on the <b>Audio Clip types</b> page.'),
    ];

    $form['min_duration'] = [
      '#title' => $this->t('Minimum Duration'),
      '#type' => 'number',
      '#default_value' => $type->getMinDuration(),
      '#description' => $this->t('Minimum duration in <b>seconds</b>, leave empty so as not to limit the minimum duration.'),
    ];

    $form['max_duration'] = [
      '#title' => $this->t('Maximum Duration'),
      '#type' => 'number',
      '#default_value' => $type->getMaxDuration(),
      '#description' => $this->t('Maximum duration in <b>seconds</b>, leave empty so as not to limit the maximum duration.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  protected function actions(array $form, FormStateInterface $form_state) {
    $actions = parent::actions($form, $form_state);
    $actions['submit']['#value'] = $this->t('Save audio clip type');
    $actions['delete']['#value'] = $this->t('Delete audio clip type');
    return $actions;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    $min_duration = $form_state->getValue('min_duration');
    $max_duration = $form_state->getValue('max_duration');

    if ($min_duration && $max_duration && $min_duration > $max_duration) {
      $form_state->setErrorByName(
        'min_duration', $this->t('Min duration cannot be larger than max duration.')
      );
    }

  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $type = $this->entity;
    $type->id = trim($type->id());
    $type->label = trim($type->label);
    $type->min_duration = trim($type->min_duration);
    $type->max_duration = trim($type->max_duration);

    $status = $type->save();

    $t_args = ['%name' => $type->label()];

    if ($status == SAVED_UPDATED) {
      $this->messenger()->addMessage($this->t('The audio clip type %name has been updated.', $t_args));
    }
    elseif ($status == SAVED_NEW) {
      $this->messenger()->addMessage($this->t('The audio clip type %name has been added.', $t_args));
      $context = array_merge($t_args, ['link' => Link::createFromRoute($this->t('View'), 'audio_clip.overview_types')->toString()]);
      $this->logger('audio_clip')->notice('Added audio clip type %name.', $context);
    }

    return $form_state->setRedirect('audio_clip.overview_types');
  }

}
