<?php

namespace Drupal\audio_video_viewer\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class AudioVideoViewerSettingsForm.
 *
 * View the audio/video files.
 */
class AudioVideoViewerSettingsForm extends ConfigFormBase {

  /**
   * The Messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  public function __construct(ConfigFactoryInterface $config_factory, MessengerInterface $messenger) {
    parent::__construct($config_factory);
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
          $container->get('config.factory'),
          $container->get('messenger')
      );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'audio_video_viewer_admin_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'audio_video_viewer.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('audio_video_viewer.settings');

    $allow_all_video_extensions = $config->get('allow_all_video_extensions');
    $allow_all_audio_extensions = $config->get('allow_all_audio_extensions');

    $supported_video_extensions = $config->get('supported_video_extensions');
    $supported_audio_extensions = $config->get('supported_audio_extensions');

    $form['allow_all_video_extensions'] = [
      '#type' => 'checkbox',
      '#description' => "Check this if you want to render all files as video.",
      '#title' => 'Allow all video extensions',
      '#default_value' => $allow_all_video_extensions,
    ];

    $form['supported_video_extensions'] = [
      '#title' => 'Supported Video Extensions',
      '#type' => 'textarea',
      '#default_value' => $supported_video_extensions,
      '#states' => [
        'visible' => [
          ':input[name=allow_all_video_extensions]' => [
            'checked' => FALSE,
          ],
        ],
      ],
    ];

    $form['allow_all_audio_extensions'] = [
      '#type' => 'checkbox',
      '#description' => "Check this if you want to render all files as audio.",
      '#title' => 'Allow all audio extensions',
      '#default_value' => $allow_all_audio_extensions,
    ];

    $form['supported_audio_extensions'] = [
      '#title' => 'Supported Audio Extensions',
      '#type' => 'textarea',
      '#default_value' => $supported_audio_extensions,
      '#states' => [
        'visible' => [
          ':input[name=allow_all_audio_extensions]' => [
            'checked' => FALSE,
          ],
        ],
      ],
    ];

    $form['reset'] = [
      '#type' => 'submit',
      '#value' => 'Restore default supported extensions',
      '#submit' => ['::resetSubmit'],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Submit the reset.
   */
  public function resetSubmit(array &$form, FormStateInterface $form_state) {
    $this->config('audio_video_viewer.settings')
      ->set('supported_video_extensions', $this->config('audio_video_viewer.settings.default')->get('supported_video_extensions'))
      ->set('supported_audio_extensions', $this->config('audio_video_viewer.settings.default')->get('supported_audio_extensions'))
      ->set('allow_all_video_extensions', $this->config('audio_video_viewer.settings.default')->get('allow_all_video_extensions'))
      ->set('allow_all_audio_extensions', $this->config('audio_video_viewer.settings.default')->get('allow_all_audio_extensions'))
      ->save();
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('audio_video_viewer.settings')
      ->set('supported_video_extensions', $form_state->getValue('supported_video_extensions'))
      ->set('supported_audio_extensions', $form_state->getValue('supported_audio_extensions'))
      ->set('allow_all_video_extensions', $form_state->getValue('allow_all_video_extensions'))
      ->set('allow_all_audio_extensions', $form_state->getValue('allow_all_audio_extensions'))
      ->save();
    parent::submitForm($form, $form_state);
  }

}
