<?php

namespace Drupal\audio_video_viewer\Plugin\Field\FieldFormatter;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\Messenger;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\file\Plugin\Field\FieldFormatter\FileFormatterBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'audio_video_viewer' formatter.
 *
 * @FieldFormatter(
 *   id = "audio_video_viewer",
 *   label = @Translation("Audio Video Viewer"),
 *   field_types = {
 *     "file"
 *   }
 * )
 */
class AudioVideoViewer extends FileFormatterBase implements ContainerFactoryPluginInterface {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The Messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The File System service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected FileSystemInterface $fileSystem;

  /**
   * The file URL generator service.
   *
   * @var \Drupal\Core\File\FileUrlGeneratorInterface
   */
  protected FileUrlGeneratorInterface $fileUrlGenerator;

  /**
   * Constructs a AudioVideoViewer instance.
   *
   * @param string $plugin_id
   *   The plugin_id for the viewer.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the viewer is associated.
   * @param array $settings
   *   The viewer settings.
   * @param string $label
   *   The viewer label display setting.
   * @param string $view_mode
   *   The view mode.
   * @param array $third_party_settings
   *   Any third party settings settings.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration interface.
   * @param \Drupal\Core\Messenger\Messenger $messenger
   *   The Messenger service.
   * @param \Drupal\Core\File\FileSystemInterface $fileSystem
   *   The file service.
   * @param \Drupal\Core\File\FileUrlGeneratorInterface $file_url_generator
   *   The file URL generator service.
   */
  public function __construct($plugin_id,
                              $plugin_definition,
                              FieldDefinitionInterface $field_definition,
                              array $settings,
                              $label,
                              $view_mode,
                              array $third_party_settings,
                              ConfigFactoryInterface $configFactory,
                              Messenger $messenger,
                              FileSystemInterface $fileSystem,
                              FileUrlGeneratorInterface $file_url_generator) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);

    $this->configFactory = $configFactory;
    $this->messenger = $messenger;
    $this->fileSystem = $fileSystem;
    $this->fileUrlGenerator = $file_url_generator;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
          $plugin_id,
          $plugin_definition,
          $configuration['field_definition'],
          $configuration['settings'],
          $configuration['label'],
          $configuration['view_mode'],
          $configuration['third_party_settings'],
          $container->get('config.factory'),
          $container->get('messenger'),
          $container->get('file_system'),
          $container->get('file_url_generator')
      );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'show_file_link' => 0,
      'show_file_size' => 0,
      'max_file_size' => 0,
      'return_empty' => 0,
      'video_original' => 1,
      'video_width' => "320px",
      'video_height' => "240px",
      'video_preload' => "metadata",
      'audio_original' => 1,
      'audio_width' => "320px",
      'audio_preload' => "metadata",
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $form = parent::settingsForm($form, $form_state);

    $marker = rand();

    $form['show_file_link'] = [
      '#title' => $this->t("Show the file's name with a link to the file."),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('show_file_link'),
    ];

    $form['show_file_size'] = [
      '#title' => $this->t("Show the file's size."),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('show_file_size'),
    ];

    $form['max_file_size'] = [
      '#title' => $this->t('Max file size'),
      '#description' => $this->t('Limit this file formatter to a maximum file size in bytes. 0 indicates unlimited.'),
      '#type' => 'textfield',
      '#attributes' => [
        ' type' => 'number',
      ],
      '#element_validate' => [[$this, 'validateSize']],
      '#required' => TRUE,
      '#default_value' => $this->getSetting('max_file_size'),
    ];

    $form['return_empty'] = [
      '#title' => $this->t('Disable all content if file format not recognized.'),
      '#description' => $this->t("If a file's content is not recognized and cannot be drawn, show nothing."),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('return_empty'),
    ];

    $form['video_label_begin'] = [
      '#type' => 'markup',
      '#markup' => t('<strong>Video</strong>'),
      '#prefix' => '<div class="video_label_settings">',
    ];

    $form['video_original'] = [
      '#type' => 'checkbox',
      '#description' => $this->t('The video display size will use the maximum available space.'),
      '#title' => $this->t('<strong>Autofit video size</strong>'),
      '#default_value' => $this->getSetting('video_original'),
      '#attributes' => [
        'class'        => [
          'video_original-' . $marker,
        ],
      ],
    ];

    $form['video_width'] = [
      '#title' => 'width',
      '#description' => $this->t('The video width should be at least one digit followed by px or %.'),
      '#type' => 'textfield',
      '#default_value' => $this->getSetting('video_width'),
      '#maxlength' => 10,
      '#states' => [
        'visible' => [
          '.video_original-' . $marker => [
            'checked'  => FALSE,
          ],
        ],
      ],
    ];

    $form['video_height'] = [
      '#title' => 'height',
      '#description' => $this->t('The video height should be at least one digit followed by px or %.'),
      '#type' => 'textfield',
      '#default_value' => $this->getSetting('video_height'),
      '#maxlength' => 10,
      '#states' => [
        'visible' => [
          '.video_original-' . $marker => [
            'checked'  => FALSE,
          ],
        ],
      ],
    ];

    $form['video_preload'] = [
      '#prefix' => '<strong>Video Preload Option</strong><div class="description">Allows you to specify the way videos should be loaded when the page loads.</div>',
      '#type' => 'radios',
      '#default_value' => $this->getSetting('video_preload'),
      '#options' => [
        'none' => 'none (the browser should NOT load the video when the page loads.)',
        'metadata' => 'metadata (the browser should load only metadata when the page loads.)',
        'auto' => 'auto (the browser should load the entire video when the page loads.)',
      ],
    ];

    $form['video_label_end'] = [
      '#type' => 'markup',
      '#markup' => '',
      '#prefix' => '</div>',
    ];

    $form['audio_label_begin'] = [
      '#type' => 'markup',
      '#markup' => t('<strong>Audio</strong>'),
      '#prefix' => '<div class="audio_label_settings">',
    ];

    $form['audio_original'] = [
      '#type' => 'checkbox',
      '#description' => $this->t('The audio display size will be determined by the browser.'),
      '#title' => $this->t('<strong>Auto set by the browser</strong>'),
      '#default_value' => $this->getSetting('audio_original'),
      '#attributes' => [
        'class'        => [
          'audio_original-' . $marker,
        ],
      ],
    ];

    $form['audio_width'] = [
      '#title' => 'width',
      '#description' => $this->t('The audio width should be at least one digit followed by px or %.'),
      '#type' => 'textfield',
      '#default_value' => $this->getSetting('audio_width'),
      '#maxlength' => 10,
      '#states' => [
        'visible' => [
          '.audio_original-' . $marker => [
            'checked'  => FALSE,
          ],
        ],
      ],
    ];

    $form['audio_preload'] = [
      '#prefix' => '<strong>Audio preload Option</strong><div class="description">Allows you to specify the way audios should be loaded when the page loads.</div>',
      '#type' => 'radios',
      '#default_value' => $this->getSetting('audio_preload'),
      '#options' => [
        'none' => 'none (the browser should NOT load the audio when the page loads.)',
        'metadata' => 'metadata (the browser should load only metadata when the page loads.)',
        'auto' => 'auto (the browser should load the entire audio when the page loads.)',
      ],
    ];

    $form['audio_label_end'] = [
      '#type' => 'markup',
      '#markup' => '',
      '#prefix' => '</div>',
    ];

    return $form;
  }

  /**
   * Validates width form element.
   *
   * @param array $element
   *   The form element to process.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param array $complete_form
   *   The complete form structure.
   */
  public static function validateSize(array &$element, FormStateInterface $form_state, array &$complete_form): void {
    $size = $element['#value'];

    if (!is_numeric($size) || $size < 0) {
      $form_state->setErrorByName("max_file_size", "The max file size for AudioVideoViewer must be greater than or equal to 0.");
    }
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary(): array {
    $summary = [];
    $settings = $this->getSettings();
    $summary[] = "Display audio/video.";
    if ($settings['return_empty'] == 1) {
      $summary[] = $this->t('Empty if file type is unrecognized.');
    }
    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode): array {
    $settings = $this->getSettings();

    // Create an empty render array.
    $elements = [];

    $config = $this->configFactory->get('audio_video_viewer.settings');
    $allow_all_video_extensions = $config->get('allow_all_video_extensions');
    $allow_all_audio_extensions = $config->get('allow_all_audio_extensions');
    $supported_video_extensions = explode(" ", $config->get('supported_video_extensions'));
    $supported_audio_extensions = explode(" ", $config->get('supported_audio_extensions'));

    // Loop through the entities we need to process and create a render
    // array for each one.
    foreach ($this->getEntitiesToView($items, $langcode) as $delta => $file) {
      $markup = '';

      $type = '';

      // Get the server file system's real path to the file.
      /** @var \Drupal\file\FileInterface $file */
      $path = $this->fileSystem->realpath($file->getFileUri());
      if ($path === FALSE) {
        $markup = '<p>' . $this->t('A server error occurred. The file could not be found.') . '</p>';
        if ($settings['return_empty'] == 1) {
          return [];
        }
      }

      // Get the user-visible file n dame.
      $mimetype = $file->getMimeType();
      $filename = $file->getFilename();
      $filepath = $this->fileUrlGenerator->generateAbsoluteString($file->getFileUri());
      // Get the file name extension and see if it is supported by
      // any of the known file formats.
      $ext = pathinfo($filename, PATHINFO_EXTENSION);

      if ($markup === '') {
        // If it is both an supported audio and video extension then
        // determine file type by mime type.
        if (
              ($allow_all_audio_extensions || ($ext && in_array($ext, $supported_audio_extensions))) &&
              ($allow_all_video_extensions || ($ext && in_array($ext, $supported_video_extensions)))
          ) {
          // It should be either video/... or audio/...
          $type = explode("/", $mimetype)[0];
          // If not audio or video, then it is unknown mimetype.
          if ($type != 'audio' && $type != 'video') {
            $markup .= '<a href="' . $filepath . '">' . $filename . '</a>';
            $this->messenger->addError('Unable to render file with AudioVideoViewer as mimeType is ' . $type . '. Mimetype should have a video or audio prefix.');
          }
        }
        // Else it's in one or the other or none.
        else {
          if ($allow_all_audio_extensions || ($ext && in_array($ext, $supported_audio_extensions))) {
            $type = 'audio';
          }

          if ($allow_all_video_extensions || ($ext && in_array($ext, $supported_video_extensions))) {
            $type = 'video';
          }
        }

        if ($markup === '' && $type !== '') {
          if ($settings['max_file_size'] > 0) {
            if ($file->getSize() > $settings['max_file_size']) {
              $markup .= '<a href="' . $filepath . '">' . $filename . '</a>';
              $this->messenger->addError('Unable to render file with AudioVideoViewer as file exceeds ' . $settings['max_file_size'] . ' bytes.');
            }
          }
        }
      }

      if ($markup === '' && $type === 'video') {
        if (!$settings['video_original']) {
          if (!$this->isValidWidthOrHeight($settings['video_width']) || !$this->isValidWidthOrHeight($settings['video_height'])) {
            $markup .= '<a href="' . $filepath . '">' . $filename . '</a>';
            $this->messenger->addError('Unable to render video file. The video width and height must be at least one digit followed by px or %');
          }
        }
      }

      if ($markup === '' && $type === 'audio') {
        if (!$settings['audio_original']) {
          if (!$this->isValidWidthOrHeight($settings['audio_width'])) {
            $markup .= '<a href="' . $filepath . '">' . $filename . '</a>';
            $this->messenger->addError('Unable to render audio file. The audio width must be at least one digit followed by px or %');
          }
        }
      }

      if ($type === '' && $markup === '') {
        $markup = '<a href="' . $filepath . '">' . $filename . '</a>';
        if ($settings['return_empty'] == 1) {
          return [];
        }
      }

      $source = [
        'src' => $filepath,
        'mimetype' => $mimetype,
      ];

      if ($type === 'audio' && $markup === '') {
        $elements[$delta] = [
          '#theme' => 'audio_tag',
          '#sources' => [$source],
          '#show_file_link' => $settings['show_file_link'],
          '#show_file_size' => $settings['show_file_size'],
          '#file_path' => $filepath,
          '#file_name' => $filename,
          '#file_size' => $file->getSize(),
          '#audio_original' => $settings['audio_original'],
          '#audio_width' => $settings['audio_width'],
          '#audio_preload' => $settings['audio_preload'],
        ];
      }
      elseif ($type === 'video' && $markup === '') {
        $elements[$delta] = [
          '#theme' => 'video_tag',
          '#sources' => [$source],
          '#show_file_link' => $settings['show_file_link'],
          '#show_file_size' => $settings['show_file_size'],
          '#file_path' => $filepath,
          '#file_name' => $filename,
          '#file_size' => $file->getSize(),
          '#video_original' => $settings['video_original'],
          '#video_width' => $settings['video_width'],
          '#video_height' => $settings['video_height'],
          '#video_preload' => $settings['video_preload'],
        ];
      }
      else {
        $elements[$delta] = [
          '#markup' => $markup,
        ];
      }
    }

    return $elements;
  }

  /**
   * The mapping table for the supported audio.
   */
  private function getSupportedAudio(): array {
    $hTable = [
      "mp3" => "audio/mpeg",
      "ogg" => "audio/ogg",
      "wav" => "audio/x-wav",
    ];

    return $hTable;
  }

  /**
   * The mapping table for the supported video.
   */
  private function getSupportedVideo(): array {
    $hTable = [
      "mp4" => "video/mp4",
      "ogg" => "video/ogg",
      "webm" => "video/webm",
    ];

    return $hTable;
  }

  /**
   * Set up the default width.
   */
  private function getDefaultWidth(): string {
    return "320px";
  }

  /**
   * Set up the default height.
   */
  private function getDefaultHeight(): string {
    return "240px";
  }

  /**
   * Check the invalid width and height.
   */
  private function isValidWidthOrHeight($width_or_height): bool {
    $px_pattern = "/^\d+[ ]*px$/";
    $percent_pattern = "/^(\d+[ ]*%|\d+[.]\d+[ ]*%)$/";

    if (preg_match($px_pattern, $width_or_height) == 1 || preg_match($percent_pattern, $width_or_height) == 1) {
      return TRUE;
    }

    return FALSE;
  }

}
