<?php

declare(strict_types=1);

namespace Drupal\auditfiles\Auditor;

use Drupal\auditfiles\AuditFilesAuditorInterface;
use Drupal\auditfiles\Event\AuditFilesDeleteFileEntityEvent;
use Drupal\auditfiles\Reference\FileEntityReference;
use Drupal\auditfiles\Services\AuditFilesConfigInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Messenger\MessengerTrait;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Service managed not used functions.
 *
 * @internal
 *   There is no extensibility promise for this class. This class may be marked
 *   as final, introducing an interface. Service decorators are recommended for
 *   extension. If extending directly, mark the original service as a service
 *   parent, and use service calls and setter injection for DI and construction
 *   as constructor is final.
 *
 *  @template R of \Drupal\auditfiles\Reference\FileEntityReference
 */
final class AuditFilesManagedNotUsed implements AuditFilesAuditorInterface, EventSubscriberInterface {

  use MessengerTrait;

  /**
   * Constructs a new AuditFilesManagedNotUsed.
   */
  final public function __construct(
    protected AuditFilesConfigInterface $auditFilesConfig,
    protected Connection $connection,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public function getReferences(): \Generator {
    $maximum_records = $this->auditFilesConfig->getReportOptionsMaximumRecords();
    $scheme = $this->auditFilesConfig->getFileSystemPath();
    $fu_query = $this->connection->select('file_usage', 'fu')->fields('fu', ['fid'])->execute()->fetchCol();
    $query = $this->connection->select('file_managed', 'fm')->fields('fm', ['fid']);
    if (!empty($fu_query)) {
      $query->condition('fm.fid', $fu_query, 'NOT IN');
    }
    if (!empty($scheme)) {
      $query->condition('fm.uri', $scheme . '://%', 'LIKE');
    }
    if ($maximum_records > 0) {
      $query->range(0, $maximum_records);
    }

    foreach ($query->execute()->fetchCol() as $fid) {
      yield FileEntityReference::create((int) $fid);
    }
  }

  /**
   * An event subscriber for deleting a file from the file_managed table.
   *
   * @internal
   *   There is no extensibility promise for this method; Use events instead.
   */
  public function listenerDeleteFileEntity(AuditFilesDeleteFileEntityEvent $event): void {
    if ($event->wasDeleted !== NULL) {
      return;
    }

    $event->reference->getFile()?->delete();
    $event->wasDeleted = TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
      AuditFilesDeleteFileEntityEvent::class => ['listenerDeleteFileEntity'],
    ];
  }

}
