<?php

namespace Drupal\auditfiles\Auditor;

use Drupal\auditfiles\AuditFilesAuditorInterface;
use Drupal\auditfiles\Batch\AuditFilesBatchProcess;
use Drupal\auditfiles\Batch\AuditFilesMergeFileReferencesBatchProcess;
use Drupal\auditfiles\Services\AuditFilesConfigInterface;
use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Messenger\MessengerTrait;

/**
 * Define all methods that used in merge file references functionality.
 *
 * @internal
 *   There is no extensibility promise for this class. This class may be marked
 *   as final, introducing an interface. Service decorators are recommended for
 *   extension. If extending directly, mark the original service as a service
 *   parent, and use service calls and setter injection for DI and construction
 *   as constructor is final.
 */
class AuditFilesMergeFileReferences implements AuditFilesAuditorInterface {

  use MessengerTrait;

  /**
   * Constructs a new AuditFilesMergeFileReferences.
   */
  final public function __construct(
    MessengerInterface $messenger,
    protected AuditFilesConfigInterface $auditFilesConfig,
    protected Connection $connection,
    protected DateFormatterInterface $dateFormatter,
    protected FileSystemInterface $fileSystem,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
    $this->setMessenger($messenger);
  }

  /**
   * {@inheritdoc}
   */
  public function getReferences(): \Generator { //file_managed
    $maximum_records = $this->auditFilesConfig->getReportOptionsMaximumRecords();
    $result_set = [];
    $query = $this->connection->select('file_managed', 'fm')
      ->fields('fm', ['fid', 'filename'])
      ->orderBy('filename', 'ASC');

    if ($maximum_records > 0) {
      $query->range(0, $maximum_records);
    }
    $files = $query->execute()->fetchAllKeyed();
    $show_single_file_names = $this->auditFilesConfig->isMergeFileReferencesShowSingleFileNames();
    foreach ($files as $file_id => $file_name) {
      if ($show_single_file_names) {
        $result_set[] = $file_name;
      }
      else {
        $query2 = 'SELECT COUNT(fid) count FROM {file_managed} WHERE filename = :filename AND fid != :fid';
        $results2 = $this->connection->query(
          $query2,
          [
            ':filename' => $file_name,
            ':fid' => $file_id,
          ]
        )->fetchField();
        if ($results2 > 0) {
          $result_set[] = $file_name;
        }
      }
    }

    yield from array_unique($result_set);
  }

  /**
   * Creates the batch for merging files.
   *
   * @param int $file_being_kept
   *   The file ID of the file to merge the others into.
   * @param array $files_being_merged
   *   The list of file IDs to be processed.
   *
   * @return \Drupal\Core\Batch\BatchBuilder
   *   The definition of the batch.
   */
  public function auditfilesMergeFileReferencesBatchMergeCreateBatch($file_being_kept, array $files_being_merged): BatchBuilder {
    $batch = (new BatchBuilder())
      ->setTitle('Merging files')
      ->setErrorMessage(\t('One or more errors were encountered processing the files.'))
      ->setFinishCallback([AuditFilesBatchProcess::class, 'finishBatch'])
      ->setProgressMessage('Completed @current of @total operations.');
    foreach ($files_being_merged as $file_id => $file_info) {
      if ($file_id != 0 && $file_id != $file_being_kept) {
        $batch->addOperation([AuditFilesMergeFileReferencesBatchProcess::class, 'create'], [
          (int) $file_being_kept,
          (int) $file_id,
        ]);
      }
    }
    return $batch;
  }

  /**
   * Deletes the specified file from the file_managed table.
   *
   * @param int $file_being_kept
   *   The file ID of the file to merge the other into.
   * @param int $file_being_merged
   *   The file ID of the file to merge.
   */
  public function auditfilesMergeFileReferencesBatchMergeProcessFile(int $file_being_kept, int $file_being_merged): void {
    $file_being_kept_results = $this->connection->select('file_usage', 'fu')
      ->fields('fu', ['module', 'type', 'id', 'count'])
      ->condition('fid', $file_being_kept)
      ->execute()
      ->fetchAll();
    if (empty($file_being_kept_results)) {
      $this->messenger()->addWarning(
        t('There was no file usage data found for the file you choose to keep. No changes were made.')
      );
      return;
    }
    $file_being_kept_data = reset($file_being_kept_results);
    $file_being_merged_results = $this->connection->select('file_usage', 'fu')
      ->fields('fu', ['module', 'type', 'id', 'count'])
      ->condition('fid', $file_being_merged)
      ->execute()
      ->fetchAll();
    if (empty($file_being_merged_results)) {
      $this->messenger()->addWarning(
        t(
          'There was an error retrieving the file usage data from the database for file ID %fid. Please check the files in one of the other reports. No changes were made for this file.',
          ['%fid' => $file_being_merged]
        )
      );
      return;
    }
    $file_being_merged_data = reset($file_being_merged_results);
    $file_being_merged_uri_results = $this->connection->select('file_managed', 'fm')
      ->fields('fm', ['uri'])
      ->condition('fid', $file_being_merged)
      ->execute()
      ->fetchAll();
    $file_being_merged_uri = reset($file_being_merged_uri_results);
    if ($file_being_kept_data->id == $file_being_merged_data->id) {
      $file_being_kept_data->count += $file_being_merged_data->count;
      // Delete the unnecessary entry from the file_usage table.
      $this->connection->delete('file_usage')
        ->condition('fid', $file_being_merged)
        ->execute();
      // Update the entry for the file being kept.
      $this->connection->update('file_usage')
        ->fields(['count' => $file_being_kept_data->count])
        ->condition('fid', $file_being_kept)
        ->condition('module', $file_being_kept_data->module)
        ->condition('type', $file_being_kept_data->type)
        ->condition('id', $file_being_kept_data->id)
        ->execute();
    }
    else {
      $this->connection->update('file_usage')
        ->fields(['fid' => $file_being_kept])
        ->condition('fid', $file_being_merged)
        ->condition('module', $file_being_merged_data->module)
        ->condition('type', $file_being_merged_data->type)
        ->condition('id', $file_being_merged_data->id)
        ->execute();
      // Update any fields that might be pointing to the file being merged.
      $this->auditfilesMergeFileReferencesUpdateFileFields($file_being_kept, $file_being_merged);
    }
    // Delete the unnecessary entries from the file_managed table.
    $this->connection->delete('file_managed')
      ->condition('fid', $file_being_merged)
      ->execute();
    // Delete the duplicate file.
    if (!empty($file_being_merged_uri->uri)) {
      $this->fileSystem->delete($file_being_merged_uri->uri);
    }
  }

  /**
   * Updates any fields that might be pointing to the file being merged.
   *
   * @param int $file_being_kept
   *   The file ID of the file to merge the other into.
   * @param int $file_being_merged
   *   The file ID of the file to merge.
   */
  public function auditfilesMergeFileReferencesUpdateFileFields($file_being_kept, $file_being_merged): void {
    $file_being_merged_fields = file_get_file_references($this->entityTypeManager->getStorage('file')->load($file_being_merged), NULL, EntityStorageInterface::FIELD_LOAD_REVISION, NULL);
    if (empty($file_being_merged_fields)) {
      return;
    }
    foreach ($file_being_merged_fields as $field_name => $field_references) {
      foreach ($field_references as $entity_type => $type_references) {
        foreach ($type_references as $id => $reference) {
          $entity = $this->entityTypeManager->getStorage($entity_type)->load($id);
          if ($entity) {
            $field_items = $entity->get($field_name)->getValue();
            $alt = $field_items[0]['alt'];
            $title = $field_items[0]['title'] ? $field_items[0]['title'] : '';
            foreach ($field_items as $item) {
              if ($item['target_id'] == $file_being_merged) {
                $file_object_being_kept = $this->entityTypeManager->getStorage('file')->load($file_being_kept);
                if (!empty($file_object_being_kept) && $entity->get($field_name)->getValue() != $file_being_kept) {
                  $entity->$field_name = [
                    'target_id' => $file_object_being_kept->id(),
                    'alt' => $alt,
                    'title' => $title,
                  ];
                }
                $entity->save();
                break;
              }
            }
          }
        }
      }
    }
  }

}
