<?php

declare(strict_types=1);

namespace Drupal\auditfiles\Auditor;

use Drupal\auditfiles\AuditFilesAuditorInterface;
use Drupal\auditfiles\Event\AuditFilesAddFileOnDiskEvent;
use Drupal\auditfiles\Event\AuditFilesDeleteFileOnDiskEvent;
use Drupal\auditfiles\Reference\DiskReference;
use Drupal\auditfiles\Services\AuditFilesConfigInterface;
use Drupal\auditfiles\Services\AuditFilesExclusions;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\file\FileInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\Mime\MimeTypeGuesserInterface;

/**
 * Define all methods used on Files not in database functionality.
 *
 * @internal
 *   There is no extensibility promise for this class. This class may be marked
 *   as final, introducing an interface. Service decorators are recommended for
 *   extension. If extending directly, mark the original service as a service
 *   parent, and use service calls and setter injection for DI and construction
 *   as constructor is final.
 *
 *  @template R of \Drupal\auditfiles\Reference\DiskReference
 */
final class AuditFilesNotInDatabase implements AuditFilesAuditorInterface, EventSubscriberInterface {

  /**
   * Constructs a new AuditFilesNotInDatabase.
   */
  final public function __construct(
    protected AuditFilesConfigInterface $auditFilesConfig,
    protected AuditFilesExclusions $exclusions,
    protected Connection $connection,
    protected FileSystemInterface $fileSystem,
    protected MimeTypeGuesserInterface $fileMimeTypeGuesser,
    protected TimeInterface $time,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public function getReferences(): \Generator {
    $exclusions = $this->exclusions->getExclusions();
    /** @var array<array{file_name: string, path_from_files_root: string}> $report_files */
    $report_files = [];
    $this->getFilesForReport('', $report_files, $exclusions);

    foreach ($report_files as $report_file) {
      // Check to see if the file is in the database.
      $file_to_check = empty($report_file['path_from_files_root'])
        ? $report_file['file_name']
        : $report_file['path_from_files_root'] . DIRECTORY_SEPARATOR . $report_file['file_name'];

      // If the file is not in the database, add to the list for displaying.
      if (!$this->isFileInDatabase($file_to_check)) {
        yield DiskReference::create($this->auditfilesBuildUri($file_to_check));
      }
    }
  }

  /**
   * An event subscriber for creating a file.
   *
   * @internal
   *   There is no extensibility promise for this method: Use events instead:
   *   set the file property in a listener with a weight before this listener.
   */
  final public function listenerCreateFile(AuditFilesAddFileOnDiskEvent $event): void {
    // Exit earlier if a file was created before this listener.
    if ($event->file !== NULL) {
      return;
    }

    $uri = $event->reference->getUri();
    $realFilenamePath = $this->fileSystem->realpath($uri);

    $file = $this->entityTypeManager
      ->getStorage('file')
      ->create();
    assert($file instanceof FileInterface);

    $file
      ->set('langcode', 'en')
      ->set('created', $this->time->getCurrentTime())
      ->setChangedTime($this->time->getCurrentTime());
    $file->setFilename(trim(basename($uri)));
    $file->setFileUri($uri);
    $file->setMimeType($this->fileMimeTypeGuesser->guessMimeType($realFilenamePath));
    $file->setSize(filesize($realFilenamePath));
    $file->setPermanent();

    $event->file = $file;
  }

  /**
   * An event subscriber for saving the file from listenerCreateFile.
   *
   * @internal
   *   There is no extensibility promise for this method: Use events instead:
   *   nullify the file property in a listener with a weight before this
   *   listener.
   */
  final public function listenerSaveFile(AuditFilesAddFileOnDiskEvent $event): void {
    if ($event->file !== NULL) {
      $event->file->save();
    }
  }

  /**
   * An event subscriber for creating a file.
   *
   * @internal
   *   There is no extensibility promise for this method; Use events instead.
   */
  final public function listenerDeleteFile(AuditFilesDeleteFileOnDiskEvent $event): void {
    if ($event->wasDeleted !== NULL) {
      return;
    }

    $event->wasDeleted = $this->fileSystem->delete(
      $event->reference->getUri(),
    );
  }

  /**
   * Get files for report.
   */
  private function getFilesForReport($path, array &$report_files, $exclusions): void {
    $file_system_stream = $this->auditFilesConfig->getFileSystemPath();
    $real_files_path = $this->fileSystem->realpath($file_system_stream . '://');
    $maximum_records = $this->auditFilesConfig->getReportOptionsMaximumRecords();
    if ($maximum_records > 0 && count($report_files) < $maximum_records) {
      $new_files = $this->auditfilesNotInDatabaseGetFiles($path, $exclusions);
      if (!empty($new_files)) {
        foreach ($new_files as $file) {
          // Check if the current item is a directory or a file.
          $item_path_check = empty($file['path_from_files_root'])
            ? $real_files_path . DIRECTORY_SEPARATOR . $file['file_name']
            : ($real_files_path . DIRECTORY_SEPARATOR . $file['path_from_files_root'] . DIRECTORY_SEPARATOR . $file['file_name']);

          if (is_dir($item_path_check)) {
            // The item is a directory, so go into it and get any files there.
            $file_path = (empty($path))
              ? $file['file_name']
              : $path . DIRECTORY_SEPARATOR . $file['file_name'];
            $this->getFilesForReport($file_path, $report_files, $exclusions);
          }
          else {
            // The item is a file, so add it to the list.
            $file['path_from_files_root'] = $this->auditfilesNotInDatabaseFixPathSeparators($file['path_from_files_root']);
            $report_files[] = $file;
          }
        }
      }
    }
  }

  /**
   * Checks if the specified file is in the database.
   *
   * @param string $filePathName
   *   The path and filename, from the "files" directory, of the file to check.
   *
   * @return bool
   *   Returns TRUE if the file was found in the database, or FALSE, if not.
   */
  private function isFileInDatabase(string $filePathName): bool {
    $uri = $this->auditfilesBuildUri($filePathName);
    $fid = $this->connection->select('file_managed', 'fm')
      ->condition('fm.uri', $uri)
      ->fields('fm', ['fid'])
      ->execute()
      ->fetchField();
    return $fid !== FALSE;
  }

  /**
   * Retrieves a list of files in the given path.
   *
   * @param string $path
   *   The path to search for files in.
   * @param string $exclusions
   *   The imploded list of exclusions from configuration.
   *
   * @return array
   *   The list of files and diretories found in the given path.
   */
  private function auditfilesNotInDatabaseGetFiles(string $path, string $exclusions): array {
    $file_system_stream = $this->auditFilesConfig->getFileSystemPath();
    $real_files_path = $this->fileSystem->realpath($file_system_stream . '://');
    // The variable to store the data being returned.
    $file_list = [];
    $scan_path = empty($path) ? $real_files_path : $real_files_path . DIRECTORY_SEPARATOR . $path;
    // Get the files in the specified directory.
    $files = array_diff(scandir($scan_path), ['..', '.']);
    foreach ($files as $file) {
      // Check to see if this file should be included.
      if ($this->auditfilesNotInDatabaseIncludeFile($real_files_path . DIRECTORY_SEPARATOR . $path, $file, $exclusions)) {
        // The file is to be included, so add it to the data array.
        $file_list[] = [
          'file_name' => $file,
          'path_from_files_root' => $path,
        ];
      }
    }
    return $file_list;
  }

  /**
   * Corrects the separators of a file system's file path.
   *
   * Changes the separators of a file path, so they are match the ones
   * being used on the operating system the site is running on.
   *
   * @param string $path
   *   The path to correct.
   *
   * @return string
   *   The corrected path.
   */
  private function auditfilesNotInDatabaseFixPathSeparators($path): string {
    $path = preg_replace('@\/\/@', DIRECTORY_SEPARATOR, $path);
    $path = preg_replace('@\\\\@', DIRECTORY_SEPARATOR, $path);
    return $path;
  }


  /**
   * Checks to see if the file is being included.
   *
   * @param string $path
   *   The complete file system path to the file.
   * @param string $file
   *   The name of the file being checked.
   * @param string $exclusions
   *   The list of files and directories not to be included in the
   *   list of files to check.
   *
   * @return bool
   *   Returns TRUE, if the path or file is being included, or FALSE,
   *   if the path or file has been excluded.
   *
   * @todo Possibly add other file streams on the system but not the one
   *   being checked to the exclusions check.
   */
  private function auditfilesNotInDatabaseIncludeFile($path, $file, string $exclusions): bool {
    if (empty($exclusions)) {
      return TRUE;
    }
    elseif (!preg_match('@' . $exclusions . '@', $file) && !preg_match('@' . $exclusions . '@', $path . DIRECTORY_SEPARATOR . $file)) {
      return TRUE;
    }
    // This path and/or file are being excluded.
    return FALSE;
  }

  /**
   * Returns the internal path to the given file.
   */
  private function auditfilesBuildUri($filePathname) {
    return sprintf('%s://%s', $this->auditFilesConfig->getFileSystemPath(), $filePathname);
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
      AuditFilesAddFileOnDiskEvent::class => [
        ['listenerCreateFile', 0],
        ['listenerSaveFile', -1000],
      ],
      AuditFilesDeleteFileOnDiskEvent::class => ['listenerDeleteFile'],
    ];
  }

}
