<?php

namespace Drupal\auditfiles\Auditor;

use Drupal\auditfiles\AuditFilesAuditorInterface;
use Drupal\auditfiles\Batch\AuditFilesBatchProcess;
use Drupal\auditfiles\Batch\AuditFilesNotOnServerBatchProcess;
use Drupal\auditfiles\Reference\FileEntityReference;
use Drupal\auditfiles\Services\AuditFilesConfigInterface;
use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Messenger\MessengerTrait;

/**
 * Managed files but file on disk is missing.
 *
 * @internal
 *   There is no extensibility promise for this class. This class may be marked
 *   as final, introducing an interface. Service decorators are recommended for
 *   extension. If extending directly, mark the original service as a service
 *   parent, and use service calls and setter injection for DI and construction
 *   as constructor is final.
 *
 * @template R of \Drupal\auditfiles\Reference\FileEntityReference
 */
class AuditFilesNotOnServer implements AuditFilesAuditorInterface {

  use MessengerTrait;

  /**
   * Constructs a new AuditFilesNotOnServer.
   */
  final public function __construct(
    MessengerInterface $messenger,
    protected AuditFilesConfigInterface $auditFilesConfig,
    protected Connection $connection,
    protected DateFormatterInterface $dateFormatter,
    protected FileSystemInterface $fileSystem,
  ) {
    $this->setMessenger($messenger);
  }

  /**
   * {@inheritdoc}
   */
  public function getReferences(): \Generator {
    $maximum_records = $this->auditFilesConfig->getReportOptionsMaximumRecords();
    $query = $this->connection->select('file_managed', 'fm');
    $query->orderBy('changed', 'DESC');
    $query->range(0, $maximum_records);
    $query->fields('fm', ['fid', 'uri']);
    /** @var array<\stdClass{uri: string, fid: string}> $results */
    $results = $query->execute()->fetchAll();
    foreach ($results as $result) {
      $target = $this->fileSystem->realpath($result->uri);
      if (!file_exists($target)) {
        yield FileEntityReference::create((int) $result->fid);
      }
    }
  }

  /**
   * Creates the batch for deleting files from the database.
   *
   * @param array $fileids
   *   The list of file IDs to be processed.
   *
   * @return \Drupal\Core\Batch\BatchBuilder
   *   The definition of the batch.
   */
  public function auditfilesNotOnServerBatchDeleteCreateBatch(array $fileids): BatchBuilder {
    $batch = (new BatchBuilder())
      ->setTitle(\t('Deleting files from the database'))
      ->setErrorMessage(\t('One or more errors were encountered processing the files.'))
      ->setFinishCallback([AuditFilesBatchProcess::class, 'finishBatch'])
      ->setProgressMessage(\t('Completed @current of @total operations.'));
    foreach ($fileids as $file_id) {
      if ($file_id != 0) {
        $batch->addOperation([
          AuditFilesNotOnServerBatchProcess::class, 'create'],
          [(int) $file_id],
        );
      }
    }
    return $batch;
  }

  /**
   * Deletes the specified file from the database.
   *
   * @param int $file_id
   *   The ID of the file to delete from the database.
   */
  public function auditfilesNotOnServerBatchDeleteProcessFile($file_id): void {
    $num_rows = $this->connection->delete('file_managed')
      ->condition('fid', $file_id)
      ->execute();
    if (empty($num_rows)) {

      $this->messenger()->addWarning(
        t(
          'There was a problem deleting the record with file ID %fid from the file_managed table. Check the logs for more information.',
          ['%fid' => $file_id]
        )
      );
    }
    else {
      $this->messenger()->addStatus(
        t(
          'Sucessfully deleted File ID : %fid from the file_managed table.',
          ['%fid' => $file_id]
        )
      );
    }
  }

}
