<?php

namespace Drupal\auditfiles\Auditor;

use Drupal\auditfiles\AuditFilesAuditorInterface;
use Drupal\auditfiles\Batch\AuditFilesBatchProcess;
use Drupal\auditfiles\Batch\AuditFilesReferencedNotUsedBatchProcess;
use Drupal\auditfiles\Reference\FileFieldReference;
use Drupal\auditfiles\Services\AuditFilesConfigInterface;
use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Messenger\MessengerTrait;

/**
 * List all methods used in referenced not used functionality.
 *
 * @internal
 *   There is no extensibility promise for this class. This class may be marked
 *   as final, introducing an interface. Service decorators are recommended for
 *   extension. If extending directly, mark the original service as a service
 *   parent, and use service calls and setter injection for DI and construction
 *   as constructor is final.
 *
 * @template R of \Drupal\auditfiles\Reference\FileFieldReference
 */
class AuditFilesReferencedNotUsed implements AuditFilesAuditorInterface {

  use MessengerTrait;

  /**
   * Constructs a new AuditFilesReferencedNotUsed.
   */
  final public function __construct(
    MessengerInterface $messenger,
    protected AuditFilesConfigInterface $auditFilesConfig,
    protected Connection $connection,
    protected EntityFieldManagerInterface $entityFieldManager,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected FileUrlGeneratorInterface $fileUrlGenerator,
  ) {
    $this->setMessenger($messenger);
  }

  /**
   * {@inheritdoc}
   *
   * @return \Generator<\Drupal\auditfiles\Reference\FileFieldReference>
   */
  public function getReferences(): \Generator {
    $files_referenced = [];
    // Get a list of all files referenced in content.
    $fields = $field_data = [];
    $fields[] = $this->entityFieldManager->getFieldMapByFieldType('image');
    $fields[] = $this->entityFieldManager->getFieldMapByFieldType('file');
    if ($fields) {
      $count = 0;
      foreach ($fields as $value) {
        foreach ($value as $table_prefix => $entity_type) {
          foreach ($entity_type as $key1 => $value1) {
            $field_data[$count]['table'] = $table_prefix . '__' . $key1;
            $field_data[$count]['column'] = $key1 . '_target_id';
            $field_data[$count]['entity_type'] = $table_prefix;
            $count++;
          }
        }
      }
      foreach ($field_data as $value) {
        $table = $value['table'];
        $column = $value['column'];
        $entity_type = $value['entity_type'];
        if ($this->connection->schema()->tableExists($table)) {
          $fu_query = $this->connection->select('file_usage', 'fu')->fields('fu', ['fid'])->execute()->fetchCol();
          $query = $this->connection->select($table, 't');
          $query->addField('t', 'entity_id');
          $query->addField('t', 'bundle');
          $query->addField('t', $column);
          if (!empty($fu_query)) {
            $query->condition($column, $fu_query, 'NOT IN');
          }
          $maximum_records = $this->auditFilesConfig->getReportOptionsMaximumRecords();
          if ($maximum_records > 0) {
            $query->range(0, $maximum_records);
          }
          $file_references = $query->execute()->fetchAll();
          foreach ($file_references as $file_reference) {
            $files_referenced[] = FileFieldReference::create(
              $table,
              $column,
              $file_reference->entity_id,
              (int) $file_reference->{$column},
              $entity_type,
              $file_reference->bundle,
            );
          }
        }
      }
    }

    yield from $files_referenced;
  }


  /**
   * Creates the batch for adding files to the file_usage table.
   *
   * @param array $referenceids
   *   The list of IDs to be processed.
   *
   * @return \Drupal\Core\Batch\BatchBuilder
   *   The definition of the batch.
   */
  public function auditfilesReferencedNotUsedBatchAddCreateBatch(array $referenceids): BatchBuilder {
    $batch = (new BatchBuilder())
      ->setTitle(\t('Adding files to the file_usage table'))
      ->setErrorMessage(\t('One or more errors were encountered processing the files.'))
      ->setFinishCallback([AuditFilesBatchProcess::class, 'finishBatch'])
      ->setProgressMessage(\t('Completed @current of @total operations.'));
    foreach ($referenceids as $reference_id) {
      if (!empty($reference_id)) {
        $batch->addOperation(
          [AuditFilesReferencedNotUsedBatchProcess::class, 'createAdd'],
          [$reference_id],
        );
      }
    }
    return $batch;
  }

  /**
   * Adds the specified file to the file_usage table.
   *
   * @param string $reference_id
   *   The ID for keeping track of the reference.
   */
  public function auditfilesReferencedNotUsedBatchAddProcessFile($reference_id): void {
    $reference_id_parts = explode('.', $reference_id);
    $data = [
      'fid' => $reference_id_parts[4],
      // @todo This is hard coded for now, but need to determine how to figure out
      // which module needs to be here.
      'module' => 'file',
      'type' => $reference_id_parts[3],
      'id' => $reference_id_parts[2],
      'count' => 1,
    ];
    // Make sure the file is not already in the database.
    $query = 'SELECT fid FROM {file_usage}
    WHERE fid = :fid AND module = :module AND type = :type AND id = :id';
    $existing_file = $this->connection->query(
      $query,
      [
        ':fid' => $data['fid'],
        ':module' => $data['module'],
        ':type' => $data['type'],
        ':id' => $data['id'],
      ]
    )->fetchAll();
    if (empty($existing_file)) {
      // The file is not already in the database, so add it.
      $this->connection->insert('file_usage')->fields($data)->execute();
    }
    else {
      $this->messenger->addError(
        t(
           'The file is already in the file_usage table (file id: "@fid", module: "@module", type: "@type", entity id: "@id").',
          [
            '@fid' => $data['fid'],
            '@module' => $data['module'],
            '@type' => $data['type'],
            '@id' => $data['id'],
          ]
        )
      );
    }
  }

  /**
   * Creates the batch for deleting file references from their content.
   *
   * @param array $referenceids
   *   The list of IDs to be processed.
   *
   * @return \Drupal\Core\Batch\BatchBuilder
   *   The definition of the batch.
   */
  public function auditfilesReferencedNotUsedBatchDeleteCreateBatch(array $referenceids): BatchBuilder {
    $batch = (new BatchBuilder())
      ->setTitle(\t('Deleting file references from their content'))
      ->setErrorMessage(\t('One or more errors were encountered processing the files.'))
      ->setFinishCallback([AuditFilesBatchProcess::class, 'finishBatch'])
      ->setProgressMessage(\t('Completed @current of @total operations.'));
    foreach ($referenceids as $reference_id) {
      if ($reference_id != '') {
        $batch->addOperation(
          [AuditFilesReferencedNotUsedBatchProcess::class, 'createDelete'],
          [$reference_id],
        );
      }
    }
    return $batch;
  }

  /**
   * Deletes the specified file from the database.
   *
   * @param string $reference_id
   *   The ID for keeping track of the reference.
   */
  public function auditfilesReferencedNotUsedBatchDeleteProcessFile($reference_id): void {
    $reference_id_parts = explode('.', $reference_id);
    $num_rows = $this->connection->delete($reference_id_parts[0])
      ->condition($reference_id_parts[1], $reference_id_parts[4])
      ->execute();
    if (empty($num_rows)) {
      $this->messenger()->addWarning(
        t(
          'There was a problem deleting the reference to file ID %fid in the %entity_type with ID %eid. Check the logs for more information.',
          [
            '%fid' => $reference_id_parts[4],
            '%entity_type' => $reference_id_parts[3],
            '%eid' => $reference_id_parts[2],
          ]
        )
      );
    }
    else {
      $this->messenger()->addStatus(
        t(
          'file ID %fid  deleted successfully.',
          [
            '%fid' => $reference_id_parts[4],
          ]
        )
      );
    }
  }

}
