<?php

namespace Drupal\auditfiles\Auditor;

use Drupal\auditfiles\AuditFilesAuditorInterface;
use Drupal\auditfiles\Batch\AuditFilesBatchProcess;
use Drupal\auditfiles\Batch\AuditFilesUsedNotManagedBatchProcess;
use Drupal\auditfiles\Reference\FileUsageReference;
use Drupal\auditfiles\Services\AuditFilesConfigInterface;
use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\Database\Connection;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Messenger\MessengerTrait;

/**
 * Form for Files used not managed functionality.
 *
 * @internal
 *   There is no extensibility promise for this class. This class may be marked
 *   as final, introducing an interface. Service decorators are recommended for
 *   extension. If extending directly, mark the original service as a service
 *   parent, and use service calls and setter injection for DI and construction
 *   as constructor is final.
 */
class AuditFilesUsedNotManaged implements AuditFilesAuditorInterface {

  use MessengerTrait;

  /**
   * Constructs a new AuditFilesUsedNotManaged.
   */
  final public function __construct(
    MessengerInterface $messenger,
    protected AuditFilesConfigInterface $auditFilesConfig,
    protected Connection $connection,
  ) {
    $this->setMessenger($messenger);
  }

  /**
   * {@inheritdoc}
   */
  public function getReferences(): \Generator {
    // Get all the file IDs in the file_usage table not in the
    // file_managed table.
    $fm_query = $this->connection->select('file_managed', 'fm')->fields('fm', ['fid'])->execute()->fetchCol();
    $query = $this->connection->select('file_usage', 'fu')
      ->fields('fu', ['fid', 'module', 'type', 'id', 'count']);
    if (!empty($fm_query)) {
      $query->condition('fu.fid', $fm_query, 'NOT IN');
    }
    $maximum_records = $this->auditFilesConfig->getReportOptionsMaximumRecords();
    if ($maximum_records > 0) {
      $query->range(0, $maximum_records);
    }

    $result = $query->execute();
    while ($row = $result->fetch()) {
      yield FileUsageReference::createFromRow($row);
    }
  }

  /**
   * Creates the batch for deleting files from the file_usage table.
   */
  public function auditfilesUsedNotManagedBatchDeleteCreateBatch(array $fileids): BatchBuilder {
    $batch = (new BatchBuilder())
      ->setTitle(\t('Deleting files from the file_usage table'))
      ->setErrorMessage(\t('One or more errors were encountered processing the files.'))
      ->setFinishCallback([AuditFilesBatchProcess::class, 'finishBatch'])
      ->setProgressMessage(\t('Completed @current of @total operations.'));
    foreach ($fileids as $file_id) {
      if ($file_id != 0) {
        $batch->addOperation(
          [AuditFilesUsedNotManagedBatchProcess::class, 'create'],
          [(int) $file_id],
        );
      }
    }
    return $batch;
  }

  /**
   * Deletes the specified file from the file_usage table.
   *
   * @param int $file_id
   *   The ID of the file to delete from the database.
   */
  public function auditfilesUsedNotManagedBatchDeleteProcessFile($file_id): void {
    $num_rows = (int) $this->connection->delete('file_usage')->condition('fid', $file_id)->execute();
    if ($num_rows === 0) {
      $this->messenger()->addWarning(\t('There was a problem deleting the record with file ID %fid from the file_usage table. Check the logs for more information.', [
        '%fid' => $file_id,
      ]));
      return;
    }

    $this->messenger()->addStatus(\t('Sucessfully deleted File ID : %fid from the file usage table.', [
      '%fid' => $file_id
    ]));
  }

}
