<?php

namespace Drupal\auditfiles\Auditor;

use Drupal\auditfiles\AuditFilesAuditorInterface;
use Drupal\auditfiles\Batch\AuditFilesBatchProcess;
use Drupal\auditfiles\Batch\AuditFilesUsedNotReferencedBatchProcess;
use Drupal\auditfiles\Services\AuditFilesConfigInterface;
use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Messenger\MessengerTrait;

/**
 * Files in usage table, but no reference from field field tables.
 *
 * @internal
 *   There is no extensibility promise for this class. This class may be marked
 *   as final, introducing an interface. Service decorators are recommended for
 *   extension. If extending directly, mark the original service as a service
 *   parent, and use service calls and setter injection for DI and construction
 *   as constructor is final.
 */
class AuditFilesUsedNotReferenced implements AuditFilesAuditorInterface {

  use MessengerTrait;

  /**
   * Constructs a new AuditFilesUsedNotReferenced.
   */
  final public function __construct(
    MessengerInterface $messenger,
    protected AuditFilesConfigInterface $auditFilesConfig,
    protected Connection $connection,
    protected EntityFieldManagerInterface $entityFieldManager,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected FileUrlGeneratorInterface $fileUrlGenerator,
  ) {
    $this->setMessenger($messenger);
  }

  /**
   * {@inheritdoc}
   */
  public function getReferences(): \Generator {//fileusage
    $maximum_records = $this->auditFilesConfig->getReportOptionsMaximumRecords();
    $query = $this->connection->select('file_usage', 'fu')
      ->fields('fu', ['fid'])
      ->distinct();
    if ($maximum_records > 0) {
      $query->range(0, $maximum_records);
    }
    $files_in_file_usage = $query->execute()->fetchCol();
    $field_data = [];
    $fields[] = $this->entityFieldManager->getFieldMapByFieldType('image');
    $fields[] = $this->entityFieldManager->getFieldMapByFieldType('file');
    $count = 0;
    foreach ($fields as $value) {
      foreach ($value as $table_prefix => $entity_type) {
        foreach ($entity_type as $key1 => $value1) {
          if ($this->entityTypeManager->getStorage($table_prefix)->getEntityType()->isRevisionable()) {
            $field_data[$count]['table'] = $table_prefix . '_revision__' . $key1;
          }
          else {
            $field_data[$count]['table'] = $table_prefix . '__' . $key1;
          }
          $field_data[$count]['column'] = $key1 . '_target_id';
          $count++;
        }
      }
    }
    foreach ($field_data as ['table' => $table, 'column' => $column]) {
      if (!$this->connection->schema()->tableExists($table)) {
        continue;
      }

      $query = "SELECT t.$column FROM {{$table}} AS t INNER JOIN {file_usage} AS f ON f.fid = t.$column";
      $result = $this->connection->query($query)->fetchCol();
      // Exclude files which are in use.
      $files_in_file_usage = array_diff($files_in_file_usage, $result);
    }

    // Return unused files.
    yield from $files_in_file_usage;
  }

  /**
   * Creates the batch for deleting files from the file_usage table.
   */
  public function auditfilesUsedNotReferencedBatchDeleteCreateBatch(array $fileids): BatchBuilder {
    $batch = (new BatchBuilder())
      ->setTitle(\t('Deleting files from the file_usage table'))
      ->setErrorMessage(\t('One or more errors were encountered processing the files.'))
      ->setFinishCallback([AuditFilesBatchProcess::class, 'finishBatch'])
      ->setProgressMessage(\t('Completed @current of @total operations.'));
    foreach ($fileids as $file_id) {
      if ($file_id != 0) {
        $batch->addOperation(
          [AuditFilesUsedNotReferencedBatchProcess::class, 'create'],
          [(int) $file_id],
        );
      }
    }
    return $batch;
  }

  /**
   * Deletes the specified file from the file_usage table.
   *
   * @param int $file_id
   *   The ID of the file to delete from the database.
   */
  public function auditfilesUsedNotReferencedBatchDeleteProcessFile($file_id): void {
    $num_rows = $this->connection->delete('file_usage')->condition('fid', $file_id)->execute();
    if (empty($num_rows)) {
      $this->messenger()->addWarning(t('There was a problem deleting the record with file ID %fid from the file_usage table. Check the logs for more information.', ['%fid' => $file_id]));
    }
    else {
      $this->messenger()->addStatus(t('Sucessfully deleted File ID : %fid from the file_usage table.', ['%fid' => $file_id]));
    }
  }

}
