<?php

declare(strict_types=1);

namespace Drupal\auditfiles\Batch;

use Drupal\auditfiles\Auditor\AuditFilesNotOnServer;
use Drupal\Component\Utility\Html;

/**
 * Process batch files.
 */
final class AuditFilesNotOnServerBatchProcess {

  /**
   * Class constructor.
   *
   * @param \Drupal\auditfiles\Auditor\AuditFilesNotOnServer $notOnServer
   *   Injected ServiceAuditFilesNotOnServer service.
   * @param int $fileId
   *   File entity ID to delete.
   */
  final protected function __construct(
    protected AuditFilesNotOnServer $notOnServer,
    protected int $fileId,
  ) {
  }

  /**
   * The batch process for deleting the file.
   *
   * Used by the Batch API to keep track of and pass data from one operation to
   * the next.
   *
   * @param int $fileId
   *   File entity ID to delete.
   * @param array $context
   *   Batch context.
   */
  public static function create($fileId, array &$context): void {
    (new static(
      static::getService(), $fileId,
    ))->dispatch($context);
  }

  /**
   * Processes file removal from file_usage not referenced in content.
   *
   * @param array $context
   *   Batch context.
   */
  protected function dispatch(array &$context): void {
    $this->notOnServer->auditfilesNotOnServerBatchDeleteProcessFile($this->fileId);
    $context['results'][] = Html::escape($this->fileId);
    $context['message'] = \t('Processed file ID %file_id.', ['%file_id' => $this->fileId]);
  }

  /**
   * Gets the related audit files service.
   */
  protected static function getService(): AuditFilesNotOnServer {
    return \Drupal::service('auditfiles.not_on_server');
  }

}
