<?php

declare(strict_types=1);

namespace Drupal\auditfiles\Batch;

use Drupal\auditfiles\Auditor\AuditFilesReferencedNotUsed;
use Drupal\Component\Utility\Html;

/**
 * Process batch files.
 */
final class AuditFilesReferencedNotUsedBatchProcess {

  /**
   * Class constructor.
   *
   * @param \Drupal\auditfiles\Auditor\AuditFilesReferencedNotUsed $referencedNotUsed
   *   Injected ServiceAuditFilesUsedNotManaged service.
   * @param string $referenceId
   *   File entity ID to delete.
   */
  final protected function __construct(
    protected AuditFilesReferencedNotUsed $referencedNotUsed,
    protected string $referenceId,
  ) {
  }

  /**
   * Batch Process for Adding a file reference.
   *
   * @param string $referenceId
   *   File entity reference ID to add.
   * @param array $context
   *   Batch context.
   */
  public static function createAdd(string $referenceId, array &$context): void {
    (new static(
      static::getService(), $referenceId,
    ))->addDispatch($context);
  }

  /**
   * Processes entity reference additions from content entities to file_managed.
   *
   * @param array $context
   *   Batch context.
   */
  protected function addDispatch(array &$context): void {
    $this->referencedNotUsed->auditfilesReferencedNotUsedBatchAddProcessFile($this->referenceId);
    $context['results'][] = Html::escape($this->referenceId);
    $context['message'] = \t('Processed file ID %file_id.', ['%file_id' => $this->referenceId]);
  }

  /**
   * Batch Process for Deleting a file reference.
   *
   * @param string $referenceId
   *   File entity reference ID to delete.
   * @param array $context
   *   Batch context.
   */
  public static function createDelete(string $referenceId, array &$context): void {
    (new static(
      static::getService(), $referenceId,
    ))->deleteDispatch($context);
  }

  /**
   * Processes entity reference deletions from content entities to file_managed.
   *
   * @param array $context
   *   Batch context.
   */
  protected function deleteDispatch(array &$context) {
    $this->referencedNotUsed->auditfilesReferencedNotUsedBatchDeleteProcessFile($this->referenceId);
    $context['results'][] = Html::escape($this->referenceId);
    $context['message'] = \t('Processed file ID %file_id.', ['%file_id' => $this->referenceId]);
  }

  /**
   * Gets the related audit files service.
   */
  protected static function getService(): AuditFilesReferencedNotUsed {
    return \Drupal::service('auditfiles.referenced_not_used');
  }

}
