<?php

declare(strict_types=1);

namespace Drupal\auditfiles\Batch;

use Drupal\auditfiles\Auditor\AuditFilesUsedNotManaged;
use Drupal\Component\Utility\Html;

/**
 * Batch Worker to remove files from file_usage not in file_managed table.
 */
final class AuditFilesUsedNotManagedBatchProcess {

  /**
   * Class constructor.
   *
   * @param \Drupal\auditfiles\Auditor\AuditFilesUsedNotManaged $usedNotManaged
   *   Injected ServiceAuditFilesUsedNotManaged service.
   * @param int $fileId
   *   File entity ID to delete.
   */
  final protected function __construct(
    protected AuditFilesUsedNotManaged $usedNotManaged,
    protected int $fileId,
  ) {
  }

  /**
   * The batch process for deleting the file feature 'used not managed'.
   *
   * @param int $fileId
   *   File entity ID to delete.
   * @param array $context
   *   Batch context.
   */
  public static function create($fileId, array &$context): void {
    (new static(
      static::getService(), $fileId,
    ))->dispatch($context);
  }

  /**
   * Processes removal of files from file_managed not in file_usage.
   *
   * @param array $context
   *   Batch context.
   */
  protected function dispatch(array &$context): void {
    $this->usedNotManaged->auditfilesUsedNotManagedBatchDeleteProcessFile($this->fileId);
    $context['results'][] = Html::escape($this->fileId);
    $context['message'] = \t('Processed file ID %file_id.', ['%file_id' => $this->fileId]);
  }

  /**
   * Gets the related audit files service.
   */
  protected static function getService(): AuditFilesUsedNotManaged {
    return \Drupal::service('auditfiles.used_not_managed');
  }

}
