<?php

declare(strict_types=1);

namespace Drupal\auditfiles\Batch;

use Drupal\auditfiles\Auditor\AuditFilesUsedNotReferenced;
use Drupal\Component\Utility\Html;

/**
 * Process batch files.
 */
final class AuditFilesUsedNotReferencedBatchProcess {

  /**
   * Class constructor.
   *
   * @param \Drupal\auditfiles\Auditor\AuditFilesUsedNotReferenced $usedNotReferenced
   *   Injected ServiceAuditFilesUsedNotManaged service.
   * @param int $fileId
   *   File entity ID to delete.
   */
  final protected function __construct(
    protected AuditFilesUsedNotReferenced $usedNotReferenced,
    protected int $fileId,
  ) {
  }

  /**
   * The batch process for process the file ID(s).
   *
   * @param int $fileId
   *   File entity ID to delete.
   * @param array $context
   *   Batch context.
   */
  public static function create(int $fileId, array &$context): void {
    (new static(
      static::getService(), $fileId,
    ))->dispatch($context);
  }

  /**
   * Processes file removal from file_usage not referenced in content.
   *
   * @param array $context
   *   Batch context.
   */
  protected function dispatch(array &$context): void {
    $this->usedNotReferenced->auditfilesUsedNotReferencedBatchDeleteProcessFile($this->fileId);
    $context['results'][] = Html::escape($this->fileId);
    $context['message'] = \t('Processed file ID %file_id.', ['%file_id' => $this->fileId]);
  }

  /**
   * Gets the related audit files service.
   */
  protected static function getService(): AuditFilesUsedNotReferenced {
    return \Drupal::service('auditfiles.used_not_referenced');
  }

}
