<?php

declare(strict_types=1);

namespace Drupal\auditfiles\Form;

use Drupal\auditfiles\Auditor\AuditFilesMergeFileReferences;
use Drupal\auditfiles\Services\AuditFilesConfigInterface;
use Drupal\Component\Render\FormattableMarkup;
use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Form\ConfirmFormHelper;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Pager\PagerManagerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Merge file references.
 *
 * Lists potential duplicate files in the file_managed table, and allows for
 * merging them into a single file.
 *
 * @internal
 *   There is no extensibility promise for this class. Use form alter hooks to
 *   make customisations.
 */
final class AuditFilesMergeFileReferencesForm extends FormBase implements AuditFilesAuditorFormInterface {

  /**
   * Constructs a new AuditFilesMergeFileReferencesForm.
   */
  final public function __construct(
    protected AuditFilesConfigInterface $auditFilesConfig,
    protected PagerManagerInterface $pagerManager,
    protected AuditFilesMergeFileReferences $mergeFileReferences,
    protected DateFormatterInterface $dateFormatter,
    protected Connection $connection,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('auditfiles.config'),
      $container->get('pager.manager'),
      $container->get('auditfiles.merge_file_references'),
      $container->get('date.formatter'),
      $container->get('database'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'mergefilereferences';
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription(): TranslatableMarkup {
    return $this->t('This action cannot be undone.');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText(): TranslatableMarkup {
    return $this->t('Confirm');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelText(): TranslatableMarkup {
    return $this->t('Cancel');
  }

  /**
   * {@inheritdoc}
   */
  public function getFormName(): string {
    return 'audit_files_merge_file_references';
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl(): Url {
    return new Url('auditfiles.audit_files_mergefilereferences');
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion(): TranslatableMarkup {
    return $this->t('Do you want to merge following record');
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    /** @var array{confirm: bool, files: string[], op: string, stage: string} $storage */
    $storage = $form_state->getStorage();
    return match (TRUE) {
      // State is "preconfirm", build form and form submit handler.
      (isset($storage['confirm']) && $storage['stage'] == 'preconfirm') => $this->buildPreConfirmForm($form, $form_state),
      // Stage is "confirm", build form and form submit handler.
      (isset($storage['confirm']) && $storage['stage'] == 'confirm') => $this->buildConfirmForm($form, $form_state),
      // The initial list form.
      default => $this->buildListForm($form, $form_state),
    };
  }

  /**
   * {@inheritdoc}
   */
  public function buildListForm(array $form, FormStateInterface $form_state): array {
    $fileNames = \iterator_to_array($this->mergeFileReferences->getReferences());
    $rows = array_reduce($fileNames, function (?array $rows, string $fileName): array {
      $rows[$fileName] = $this->fileRow($fileName);
      return $rows;
    });
    // Set up the pager.
    if (!empty($rows)) {
      $items_per_page = $this->auditFilesConfig->getReportOptionsItemsPerPage();
      if (!empty($items_per_page)) {
        $current_page = $this->pagerManager->createPager(count($rows), $items_per_page)->getCurrentPage();
        // Break the total data set into page sized chunks.
        $pages = array_chunk($rows, $items_per_page, TRUE);
      }
    }
    // Setup the record count and related messages.
    $maximum_records = $this->auditFilesConfig->getReportOptionsMaximumRecords();
    if (!empty($rows)) {
      if ($maximum_records > 0) {
        $file_count_message = 'Found at least @count files in the file_managed table with duplicate file names.';
      }
      else {
        $file_count_message = 'Found @count files in the file_managed table with duplicate file names.';
      }
      $form_count = $this->formatPlural(count($rows), 'Found 1 file in the file_managed table with a duplicate file name.', $file_count_message);
    }
    else {
      $form_count = $this->t('Found no files in the file_managed table with duplicate file names.');
    }
    $form['filter']['single_file_names']['auditfiles_merge_file_references_show_single_file_names'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show files without duplicate names'),
      '#default_value' => $this->auditFilesConfig->isMergeFileReferencesShowSingleFileNames(),
      '#suffix' => '<div>' . $this->t("Use this button to reset this report's variables and load the page anew.") . '</div>',
    ];
    $form['files'] = [
      '#type' => 'tableselect',
      '#header' => [
        'filename' => [
          'data' => t('Name'),
        ],
        'references' => [
          'data' => t('File IDs using the filename'),
        ],
      ],
      '#empty' => $this->t('No items found.'),
      '#prefix' => '<div><em>' . $form_count . '</em></div>',
    ];
    if (!empty($rows) && !empty($pages)) {
      $form['files']['#options'] = $pages[$current_page];
    }
    elseif (!empty($rows)) {
      $form['files']['#options'] = $rows;
    }
    else {
      $form['files']['#options'] = [];
    }
    if (!empty($rows)) {
      $form['actions'] = ['#type' => 'actions'];
      $form['actions']['submit'] = [
        '#type' => 'submit',
        '#value' => $this->t('Merge selected items'),
      ];
      $form['pager'] = ['#type' => 'pager'];
    }
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfirmForm(array $form, FormStateInterface $form_state): array {
    $date_format = $this->auditFilesConfig->getReportOptionsDateFormat();
    $header = [
      'origname' => [
        'data' => $this->t('Filename'),
      ],
      'fileid' => [
        'data' => $this->t('File ID'),
      ],
      'fileuri' => [
        'data' => $this->t('URI'),
      ],
      'filesize' => [
        'data' => $this->t('Size'),
      ],
      'timestamp' => [
        'data' => $this->t('Time uploaded'),
      ],
    ];
    $files = [];
    $values = $form_state->getValue('files_being_merged');
    foreach ($values as $file_id) {
      if (!empty($file_id)) {
        $query = $this->connection->select('file_managed', 'fm');
        $query->condition('fm.fid', $file_id);
        $query->fields('fm', [
          'fid',
          'uid',
          'filename',
          'uri',
          'filemime',
          'filesize',
          'created',
          'status',
        ]);
        $results = $query->execute()->fetchAll();
        $file = $results[0];
        if (!empty($file)) {
          $files[$file_id] = [
            'origname' => $file->filename,
            'fileid' => $file_id,
            'fileuri' => $file->uri,
            'filesize' => \number_format((float) $file->filesize),
            'timestamp' => $this->dateFormatter->format($file->created, $date_format),

          ];
        }
        else {
          $this->messenger()->addStatus(
            $this->t('A file object was not found for file ID @fid.', ['@fid' => $file_id])
          );
        }
      }
      else {
        unset($form_state->getValue('files_being_merged')[$file_id]);
      }
    }
    // Save the selected items for later use.
    $storage['files_being_merged'] = $files;
    $form_state->setStorage($storage);
    $form['file_being_kept'] = [
      '#type' => 'tableselect',
      '#header' => $header,
      '#options' => $files,
      '#default_value' => key($files),
      '#empty' => $this->t('No items found.'),
      '#multiple' => FALSE,
    ];
    $form['#title'] = $this->t('Which file will be the one the others are merged into?');
    $form['#attributes']['class'][] = 'confirmation';
    $form['actions'] = [
      '#type' => 'actions',
    ];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->getConfirmText(),
      '#button_type' => 'primary',
      '#submit' => ['::confirmSubmissionHandlerFileMerge'],
    ];
    $form['actions']['cancel'] = ConfirmFormHelper::buildCancelLink($this, $this->getRequest());
    if (!isset($form['#theme'])) {
      $form['#theme'] = 'confirm_form';
    }
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function buildPreConfirmForm(array $form, FormStateInterface $form_state): array {
    $date_format = $this->auditFilesConfig->getReportOptionsDateFormat();
    $header = [
      'filename' => [
        'data' => $this->t('Filename'),
      ],
      'fileid' => [
        'data' => $this->t('File ID'),
      ],
      'fileuri' => [
        'data' => $this->t('URI'),
      ],
      'filesize' => [
        'data' => $this->t('Size'),
      ],
      'timestamp' => [
        'data' => $this->t('Time uploaded'),
      ],
    ];
    $files = [];
    $values = $form_state->getValue('files');
    foreach ($values as $file_name) {
      if (!empty($file_name)) {
        $query = 'SELECT fid FROM {file_managed} WHERE filename = :file_name ORDER BY uri ASC';
        $results = $this->connection->query($query, [':file_name' => $file_name])->fetchAll();
        if (!empty($results)) {
          foreach ($results as $result) {
            $query = $this->connection->select('file_managed', 'fm');
            $query->condition('fm.fid', $result->fid);
            $query->fields('fm', [
              'fid',
              'uid',
              'filename',
              'uri',
              'filemime',
              'filesize',
              'created',
              'status',
            ]);
            $results = $query->execute()->fetchAll();
            $file = $results[0];
            if (!empty($file)) {
              $files[$result->fid] = [
                'filename' => $file->filename,
                'fileid' => $file->fid,
                'fileuri' => $file->uri,
                'filesize' => \number_format((float) $file->filesize),
                'timestamp' => $this->dateFormatter->format($file->created, $date_format),
              ];
            }
            else {
              $this->messenger()->addStatus(
                $this->t('A file object was not found for file ID @fid.', ['@fid' => $result->fid])
              );
            }
          }
        }
      }
      else {
        unset($form_state->getValue('files')[$file_name]);
      }
    }
    $form['files_being_merged'] = [
      '#type' => 'tableselect',
      '#header' => $header,
      '#options' => $files,
      '#empty' => $this->t('No items found.'),
    ];
    if (!empty($files)) {
      $form['actions'] = [
        '#type' => 'actions',
        'submit' => [
          '#type' => 'submit',
          '#value' => $this->t('Next step'),
          '#submit' => ['::submissionHandlerMergeRecordPreConfirm'],
        ],
      ];
    }
    return $form;
  }

  /**
   * Form validation.
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $storage = $form_state->getStorage();
    if (isset($storage['confirm'])) {
      if ($storage['stage'] == 'preconfirm') {
        $counter = 0;
        foreach ($form_state->getValue('files_being_merged') as $file) {
          if (!empty($file)) {
            $counter++;
          }
        }
        if ($counter < 2) {
          $form_state->setErrorByName('files_being_merged', $this->t('At least two file IDs must be selected in order to merge them.'));
        }
      }
    }
  }

  /**
   * Submit form handler for Merge Records.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // @todo this should be moved to configuration form.
    $this->configFactory()->getEditable('auditfiles.settings')
      ->set('auditfiles_merge_file_references_show_single_file_names', $form_state->getValue('auditfiles_merge_file_references_show_single_file_names'))
      ->save();
    if (!empty($form_state->getValue('files'))) {
      foreach ($form_state->getValue('files') as $file_id) {
        if (!empty($file_id)) {
          $storage = [
            'files' => $form_state->getValue('files'),
            'confirm' => TRUE,
            'stage' => 'preconfirm',
          ];
          $form_state->setStorage($storage);
          $form_state->setRebuild();
        }
      }
      if (!isset($storage)) {
        $this->messenger()->addStatus(
          $this->t('At least one file name must be selected in order to merge the file IDs. No changes were made.')
        );
      }
    }
  }

  /**
   * Preconfirm form submission.
   */
  public function submissionHandlerMergeRecordPreConfirm(array &$form, FormStateInterface $form_state): void {
    if (!empty($form_state->getValue('files_being_merged'))) {
      foreach ($form_state->getValue('files_being_merged') as $file_id) {
        if (!empty($file_id)) {
          $storage = [
            'files_being_merged' => $form_state->getValue('files_being_merged'),
            'confirm' => TRUE,
            'stage' => 'confirm',
          ];
          $form_state->setStorage($storage);
          $form_state->setRebuild();
        }
      }
    }
  }

  /**
   * Confirm form submission.
   */
  public function confirmSubmissionHandlerFileMerge(array &$form, FormStateInterface $form_state): void {
    $storage = $form_state->getStorage();
    \batch_set($this->mergeFileReferences->auditfilesMergeFileReferencesBatchMergeCreateBatch($form_state->getValue('file_being_kept'), $storage['files_being_merged'])->toArray());
    unset($storage['stage']);
  }

  /**
   * Retrieves information about an individual file from the database.
   *
   * @param int $file_name
   *   The ID of the file to prepare for display.
   *
   * @return array
   *   The row for the table on the report, with the file's
   *   information formatted for display.
   */
  protected function fileRow($file_name): array {
    $fid_query = 'SELECT fid FROM {file_managed} WHERE filename = :filename';
    $fid_results = $this->connection->query($fid_query, [':filename' => $file_name])->fetchAll();
    if (count($fid_results) > 0) {
      $references = '<ul>';
      foreach ($fid_results as $fid_result) {
        $query = $this->connection->select('file_managed', 'fm');
        $query->condition('fm.fid', $fid_result->fid);
        $query->fields('fm', [
          'fid',
          'uid',
          'filename',
          'uri',
          'filemime',
          'filesize',
          'created',
          'status',
        ]);
        $results = $query->execute()->fetchAll();
        $file = $results[0];
        $references .= '<li>' . t('<strong>Fid: </strong> %id , <strong>Name : </strong> %file , <strong>File URI: </strong> %uri , <strong>Date: </strong> %date',
            [
              '%id' => $file->fid,
              '%file' => $file->filename,
              '%uri' => $file->uri,
              '%date' => $this->dateFormatter->format($file->created, $this->auditFilesConfig->getReportOptionsDateFormat()),
            ]
          ) . '</li>';
      }
      $references .= '</ul>';
      $usage = new FormattableMarkup($references, []);
      return [
        'filename' => $file_name,
        'references' => $usage,
      ];
    }
  }

}
